import { useState } from 'react';
import bread from '@/assets/breadcrumb.png';
import './sidebar.css';
import { Button } from '../Elements';
import 'react-responsive-modal/styles.css';
import { Modal } from 'react-responsive-modal';

import UserAuth from '@/components/Authentication/UserAuth';
import { useNavigate } from 'react-router-dom';
import { Form, InputField } from '@/components/Form';
import { axios } from '@/lib/axios';
import { useNotificationStore } from '@/stores/notifications';
import { z } from 'zod';
import { useAuth } from '@/lib/auth';
import i18next from 'i18next';

type ForSaleBread = {
  head?: string;
  button?: string;
  buttons?: string;
  photo?: string;
  linkAddress?: string;
  propertyType?: string;
};

const schema = z.object({
  name: z.string().min(1, 'Please enter first name'),
  mobile_number: z
    .string()
    .min(1, 'Please enter phone number')
    .regex(/^[-0-9+]+$/, 'Phone should contain numbers only!'),
  email: z.string().email('Please enter a valid email address'),
  property_type: z.string().min(1, 'Select property Type').optional(),
});

type HouseRentFormValues = {
  name: string;
  mobile_number: string;
  email: string;
  property_type: string;
};

export const ForSaleBread = ({
  head,
  button,
  buttons,
  photo,
  linkAddress,
  propertyType,
}: ForSaleBread) => {
  const auth = useAuth();
  const userEmail = auth?.user?.email;
  const userName = auth?.user?.first_name;
  const [open, setOpen] = useState(false);
  const onOpenModal = () => setOpen(true);
  const onCloseModal = () => setOpen(false);

  const [open2, setOpen2] = useState(false);
  const onOpenModal2 = () => setOpen2(true);
  const onCloseModal2 = () => setOpen2(false);

  const handleFormModal = () => {
    setOpen(false);
    setOpen2(true);
  };
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (values: HouseRentFormValues) => {
    try {
      setLoading(true);
      if (propertyType === '1') {
        values.property_type = 'InvestmentProject';
      } else if (propertyType === '2') {
        values.property_type = 'BusinessSale';
      }

      axios.post('/customer-api/send-property-requests', values);
      useNotificationStore.getState().addNotification({
        type: 'success',
        title: 'Success',
        message: 'Request sent successfully..',
      });
      setLoading(false);
      onCloseModal2();
    } finally {
      setLoading(false);
      onCloseModal2();
    }
  };

  const navigate = useNavigate();
  return (
    <div className="breadcrumb-box2" style={{ backgroundImage: 'url(' + bread + ')' }}>
      <div className="overlay-box">
        <div className="container">
          <div className="breadcrumb-content2 row">
            <div className="col-12 col-md-5 left">
              <span className="sideSq">
                <img src={photo} alt="girlpic" />
              </span>
            </div>
            <div className="col-12 col-md-7 right">
              <h3>{head}</h3>
              <div className="row fool">
                <div className="col-12 col-md-6 rights ">
                  {button ? (
                    <UserAuth click={() => navigate(linkAddress ?? '/')}>
                      <Button className="btn-rounded">{button}</Button>
                    </UserAuth>
                  ) : null}
                </div>
                <div className="col-12 col-md-6 lefts">
                  {buttons ? (
                    <Button className="btn-rounded" onClick={onOpenModal}>
                      {buttons}
                    </Button>
                  ) : (
                    ' '
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
      <Modal open={open} onClose={onCloseModal} center>
        <section className="model-datas">
          <div className="row">
            <h3 className="sselect">Select one</h3>
            <div className="col-12 col-sm-6 rightAlignButton ">
              <UserAuth
                click={() =>
                  navigate(
                    propertyType === '2' ? '/user/create-business' : linkAddress ? linkAddress : '/'
                  )
                }
              >
                <Button>Do it by yourself</Button>
              </UserAuth>
            </div>
            <div className="col-12 col-sm-6 leftAlignButton">
              <Button onClick={() => handleFormModal()}> Do it by Reeipo Team</Button>
            </div>
          </div>
        </section>
      </Modal>

      <Modal open={open2} onClose={onCloseModal2} center>
        <section className="add-models-form">
          <div className="row">
            <h3 className="sselect">Details for Reeipo Team</h3>

            <Form<HouseRentFormValues, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: {
                  email: userEmail,
                  name: userName,
                },
              }}
            >
              {({ register, formState }) => (
                <div className=" mb-5">
                  <div className="row mt-4">
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label={i18next.t('First name')}
                        error={formState.errors['name']}
                        registration={register('name')}
                        placeholder={i18next.t('First name')}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label={i18next.t('Email')}
                        error={formState.errors['email']}
                        registration={register('email')}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="number"
                        label="Contact Number"
                        error={formState.errors['mobile_number']}
                        registration={register('mobile_number')}
                      />
                    </div>
                    {propertyType === '1' && (
                      <div className="col-6 mt-4">
                        <InputField
                          type="number"
                          disabled
                          label="Investment for project"
                          error={formState.errors['property_type']}
                          registration={register('property_type')}
                        />
                      </div>
                    )}
                    {propertyType === '2' && (
                      <div className="col-6 mt-4">
                        <InputField
                          type="number"
                          disabled
                          label={i18next.t('Business For Sale')}
                          error={formState.errors['property_type']}
                          registration={register('property_type')}
                        />
                      </div>
                    )}
                    <div className="row mt-5">
                      <div className="col-6 col-sm-6 rightAlignButton">
                        <Button onClick={onCloseModal2}> Cancel</Button>
                      </div>
                      <div className="col-6 col-sm-6 leftAlignButton">
                        <Button isLoading={loading} type="submit">
                          {' '}
                          Send
                        </Button>
                      </div>
                    </div>
                  </div>
                </div>
              )}
            </Form>
          </div>
        </section>
      </Modal>
    </div>
  );
};

export default ForSaleBread;
