import { Button } from '@/components/Elements';
import { Form, InputField } from '@/components/Form';
import * as z from 'zod';
import { useState } from 'react';
import { useNotificationStore } from '@/stores/notifications';
import { changePassword } from '../../api/changePassword';
import { useAuth } from '@/lib/auth';
import { useNavigate } from 'react-router-dom';
import { ContentLayout } from '@/components/Layout';
import i18next from 'i18next';

const schema = z
  .object({
    old_password: z
      .string()
      .min(1, 'Please enter password')
      .regex(
        /^.*(?=.{3,})(?=.*[a-zA-Z])(?=.*[0-9])(?=.*[\d\x])(?=.*[!@$#%]).*$/,
        'Password should be a combination of uppercase alphabets, numbers and special characters'
      ),
    new_password: z
      .string()
      .min(1, 'Please enter password')
      .regex(
        /^.*(?=.{3,})(?=.*[a-zA-Z])(?=.*[0-9])(?=.*[\d\x])(?=.*[!@$#%]).*$/,
        'Password should be a combination of uppercase alphabets, numbers and special characters'
      ),
    confirm_password: z.string().min(1, 'Please enter confirm password'),
  })
  .superRefine(({ confirm_password, new_password }, ctx) => {
    if (confirm_password !== new_password) {
      ctx.addIssue({
        path: ['confirm_password'],
        code: 'custom',
        message: 'The passwords did not match',
      });
    }
  });

type ResetPasswordValues = {
  old_password: string;
  new_password: string;
  confirm_password: string;
};

export const AdminChangePasswordForm = () => {
  const navigate = useNavigate();
  const auth = useAuth();
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (values: ResetPasswordValues) => {
    try {
      setLoading(true);

      await changePassword(values);
      useNotificationStore.getState().addNotification({
        type: 'success',
        title: 'Success',
        message: 'Your password has been changed Successfully',
      });
      localStorage.removeItem('eipo_token');
      auth.logout();
    } catch (e: any) {
      const message = e.response?.data?.message || e.message;
      useNotificationStore.getState().addNotification({
        type: 'error',
        title: 'Error',
        message,
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <ContentLayout title="Change Password">
      <div className="p-5">
        <h1 className="mb-0 mt-3">Reset Password</h1>
        <div className="mt-4">
          <Form<ResetPasswordValues, typeof schema> onSubmit={handleSubmit} schema={schema}>
            {({ register, formState }) => (
              <>
                <div className="row align-items-center">
                  <div className="col-4">
                    <h5>Current Password</h5>
                    <InputField
                      type="password"
                      label="Current Password"
                      error={formState.errors['old_password']}
                      registration={register('old_password')}
                      placeholder="Current Password"
                    />
                  </div>
                  <div className="col-4">
                    <h5>New Password</h5>
                    <InputField
                      type="password"
                      label="New Password"
                      error={formState.errors['new_password']}
                      registration={register('new_password')}
                      placeholder="New Password"
                    />
                  </div>
                  <div className="col-4">
                    <h5>{i18next.t('Confirm Password')}</h5>
                    <InputField
                      type="password"
                      label={i18next.t('Confirm Password')}
                      error={formState.errors['confirm_password']}
                      registration={register('confirm_password')}
                      placeholder={i18next.t('Confirm Password')}
                    />
                  </div>
                </div>
                <div className="flex gap-2">
                  <Button
                    isLoading={loading}
                    variant="outline"
                    size="sm"
                    className="mt-2"
                    onClick={() => navigate(-1)}
                  >
                    {i18next.t(`Cancel`)}
                  </Button>
                  <Button
                    isLoading={loading}
                    type="submit"
                    variant="primary"
                    size="sm"
                    className="mt-2"
                  >
                    Update
                  </Button>
                </div>
              </>
            )}
          </Form>
        </div>
      </div>
    </ContentLayout>
  );
};
