import React, { useEffect, useState } from 'react';
import { Button, Table } from '@/components/Elements';
import { Business } from '@/types';
import TableLoader from '@/components/Elements/Loader/TableLoader';
import { useBusiness } from '@/api/business/getBusiness';
import { ConfirmDialogWithButton } from '@/components/Elements/ConfirmationDialog/ConfirmDialogWithButton';
import { changeStatus } from '@/api/business/changeStatus';
import { useNotificationStore } from '@/stores/notifications';
import { useNavigate } from 'react-router-dom';

const BusinessList = ({
  page,
  changePage,
}: {
  page: string;
  changePage: (page: string) => void;
}) => {
  const { data, refetch, isLoading, isFetching } = useBusiness({ page });
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();

  useEffect(() => {
    refetch();
  }, [page, refetch]);

  const changeStatusAction = async (id: string, accept: boolean) => {
    try {
      setLoading(true);
      await changeStatus(id, accept);
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'You have approved this house request successfully!',
      });
      refetch();
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
      {isLoading || isFetching ? (
        <TableLoader />
      ) : (
        <Table<Business>
          page={parseInt(page)}
          changePage={changePage}
          total={data?.total ?? 0}
          data={data?.data ?? []}
          columns={[
            { title: 'Company Name', field: 'company_name' },
            { title: 'Company Telephone', field: 'company_telephone' },
            { title: 'Category', field: 'company_category' },
            { title: 'Purchase Price', field: 'purchase_price' },
            { title: 'Total Turnover', field: 'total_turnover' },
            { title: 'Status', field: 'status' },
            {
              title: '',
              field: 'status',
              Cell({ entry: { status, id } }) {
                return (
                  <>
                    {status === 'Pending' && (
                      <ConfirmDialogWithButton
                        triggerButton={<Button variant="outlineDark">Approve</Button>}
                        confirmButton={
                          <Button
                            isLoading={loading}
                            onClick={() => changeStatusAction(id, true)}
                            variant="primary"
                          >
                            Approve
                          </Button>
                        }
                        title="Confirm"
                        icon="info"
                        body="Are you sure you want to approve this house request?"
                      />
                    )}
                  </>
                );
              },
            },
            {
              title: 'Action',
              field: 'status',
              Cell({ entry: { id } }) {
                return (
                  <>
                    <span
                      role="button"
                      tabIndex={-1}
                      onKeyDown={() => navigate(`/admin/business-listings/${id}`)}
                      onClick={() => navigate(`/admin/business-listings/${id}`)}
                      className="icon me-2"
                      title="View"
                    >
                      <i className="fa-regular fa-eye" />
                    </span>
                  </>
                );
              },
            },
          ]}
        />
      )}
    </>
  );
};

export default BusinessList;
