import React, { useEffect, useState } from 'react';
import { Button, Table } from '@/components/Elements';
import { Property } from '@/types';
import TableLoader from '@/components/Elements/Loader/TableLoader';
import { ConfirmDialogWithButton } from '@/components/Elements/ConfirmationDialog/ConfirmDialogWithButton';
import { changeStatus } from '@/api/property/changeStatus';
import { useNotificationStore } from '@/stores/notifications';
import { useNavigate } from 'react-router-dom';
import { useProperty } from '@/api/property/getProperty';

const PropertyList = ({
  page,
  changePage,
}: {
  page: string;
  changePage: (page: string) => void;
}) => {
  const { data, refetch, isLoading, isFetching } = useProperty({ page });
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();
  const [moddedProperty, setModdedProperty] = useState<any>([]);

  useEffect(() => {
    let propertiesMod: any = [];

    propertiesMod = data?.data.map((item) => {
      const property: any = {};
      property.id = item.id;
      property.property_type = item.property_type;
      property.title = item.title;
      property.category = item.category;
      property.subcategory = item.subcategory;
      property.image = item.image;
      property.customer = item.customer;
      property.status = item.status == 1 ? 'Approved' : 'Pending';

      return property;
    });
    setModdedProperty(propertiesMod);
  }, [data]);

  useEffect(() => {
    refetch();
  }, [page, refetch]);

  const changeStatusAction = async (id: string, accept: boolean) => {
    try {
      setLoading(true);
      await changeStatus(id, accept);
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'You have approved this listing successfully!',
      });
      refetch();
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
      {isLoading || isFetching ? (
        <TableLoader />
      ) : (
        <Table<Property>
          page={parseInt(page)}
          changePage={changePage}
          total={data?.total ?? 0}
          data={moddedProperty ?? []}
          columns={[
            { title: 'Title', field: 'title' },
            { title: 'Customer', field: 'customer' },
            { title: 'Type', field: 'property_type' },
            { title: 'Status', field: 'status' },
            {
              title: '',
              field: 'status',
              Cell({ entry: { status, id } }) {
                return (
                  <>
                    {status === 'Pending' && (
                      <ConfirmDialogWithButton
                        triggerButton={<Button variant="outlineDark">Approve</Button>}
                        confirmButton={
                          <Button
                            isLoading={loading}
                            onClick={() => changeStatusAction(id, true)}
                            variant="primary"
                          >
                            Approve
                          </Button>
                        }
                        title="Confirm"
                        icon="info"
                        body="Are you sure you want to approve this property?"
                      />
                    )}
                  </>
                );
              },
            },
            {
              title: 'Action',
              field: 'status',
              Cell({ entry: { id } }) {
                return (
                  <>
                    <span
                      role="button"
                      tabIndex={-1}
                      onKeyDown={() => navigate(`/admin/property/${id}`)}
                      onClick={() => navigate(`/admin/property/${id}`)}
                      className="icon me-2"
                      title="View"
                    >
                      <i className="fa-regular fa-eye" />
                    </span>
                  </>
                );
              },
            },
          ]}
        />
      )}
    </>
  );
};

export default PropertyList;
