import React, { useEffect, useState } from 'react';
import { Button, Table } from '@/components/Elements';
import { Property } from '@/types';
import TableLoader from '@/components/Elements/Loader/TableLoader';
import { ConfirmDialogWithButton } from '@/components/Elements/ConfirmationDialog/ConfirmDialogWithButton';
import { useNotificationStore } from '@/stores/notifications';
import { useNavigate } from 'react-router-dom';
import { changeStatus } from '@/api/housing/changeStatus';
import { usePropertySaleRent } from '@/features/admin/api/getPropertySaleRentList';

const PropertySaleRentList = ({
  page,
  changePage,
}: {
  page: string;
  changePage: (page: string) => void;
}) => {
  const { data, refetch, isLoading, isFetching } = usePropertySaleRent({ page });
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();

  useEffect(() => {
    refetch();
  }, [page, refetch]);

  const changeStatusAction = async (id: string, accept: boolean) => {
    try {
      setLoading(true);
      await changeStatus(id, accept);
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'You have approved this Property successfully!',
      });
      refetch();
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
      {isLoading || isFetching ? (
        <TableLoader />
      ) : (
        <Table<Property>
          page={parseInt(page)}
          changePage={changePage}
          total={data?.total ?? 0}
          data={data?.data ?? []}
          columns={[
            { title: 'Title', field: 'title' },
            { title: 'Status', field: 'status' },
            {
              title: '',
              field: 'status',
              Cell({ entry: { status, id } }) {
                return (
                  <>
                    {status === 'Pending' && (
                      <ConfirmDialogWithButton
                        triggerButton={<Button variant="outlineDark">Approve</Button>}
                        confirmButton={
                          <Button
                            isLoading={loading}
                            onClick={() => changeStatusAction(id, true)}
                            variant="primary"
                          >
                            Approve
                          </Button>
                        }
                        title="Confirm"
                        icon="info"
                        body="Are you sure you want to approve this house request?"
                      />
                    )}
                  </>
                );
              },
            },
            {
              title: 'Action',
              field: 'status',
              Cell({ entry: { id } }) {
                return (
                  <>
                    <span
                      role="button"
                      tabIndex={-1}
                      onKeyDown={() => navigate(`/admin/property-Sale-Rent-detail/${id}`)}
                      onClick={() => navigate(`/admin/property-Sale-Rent-detail/${id}`)}
                      className="icon me-2"
                      title="View"
                    >
                      <i className="fa-regular fa-eye" />
                    </span>
                  </>
                );
              },
            },
          ]}
        />
      )}
    </>
  );
};

export default PropertySaleRentList;
