import React, { useEffect, useState } from 'react';
import { Button, Table } from '@/components/Elements';
import { User } from '@/types';
import TableLoader from '@/components/Elements/Loader/TableLoader';
import { useAgentRequests } from '@/api/agentRequests/getAgentRequests';
import { loginAsUser } from '@/api/agentRequests/loginAsUser';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useNotificationStore } from '@/stores/notifications';
import storage from '@/utils/storage';
import { ConfirmDialogWithButton } from '@/components/Elements/ConfirmationDialog/ConfirmDialogWithButton';
import i18next from 'i18next';

const TeamAgentList = ({
  page,
  changePage,
}: {
  page: string;
  changePage: (page: string) => void;
}) => {
  const { data, refetch, isLoading } = useAgentRequests({ page });

  const [apiResponse, setApiResponse] = useState();
  const [loading, setLoading] = useState<boolean>(false);

  const adminToken = storage.getToken();
  const accessToken = apiResponse?.access;
  const loginType = apiResponse?.login_by;
  useEffect(() => {
    if (apiResponse) {
      localStorage.setItem('adminToken', adminToken);
      localStorage.setItem('accessToken', accessToken);
      localStorage.setItem('loginType', loginType);
    }
  }, [apiResponse]);

  useEffect(() => {
    refetch();
  }, [page, refetch]);
  const handleLogins = async (customerId: any) => {
    try {
      setLoading(true);
      const response = await loginAsUser(customerId);
      setApiResponse(response?.data);
      storage.setToken(response?.data?.access);
      setLoading(false);
      location.href = '/user/add';
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'Logged in as Admin Agent!',
      });
    } catch {
      setLoading(false);
    }
  };

  return (
    <>
      <BackdropLoader open={loading} />
      {isLoading ? (
        <TableLoader />
      ) : (
        <Table<User>
          page={parseInt(page)}
          changePage={changePage}
          total={data?.total ?? 0}
          data={data?.data ?? []}
          columns={[
            { title: `${i18next.t('Name')}`, field: 'name' },
            { title: `${i18next.t('Email')}`, field: 'email' },
            { title: `${i18next.t('Phone number')}`, field: 'mobile_number' },
            { title: `${i18next.t('Property Type')}`, field: 'property_type' },

            {
              title: 'Action',
              field: 'action',
              Cell({ entry: { customer_id } }) {
                return (
                  <>
                    <ConfirmDialogWithButton
                      triggerButton={
                        <span
                          role="button"
                          tabIndex={-1}
                          className="icon me-2"
                          title="Login as User"
                        >
                          <i className="fa-solid fa-right-to-bracket"></i>
                        </span>
                      }
                      confirmButton={
                        <Button
                          isLoading={loading}
                          onClick={() => handleLogins(customer_id)}
                          variant="primary"
                        >
                          {i18next.t('Yes, Login')}
                        </Button>
                      }
                      title="Confirm"
                      icon="info"
                      body="Are you sure you want to Login as User ?"
                    />
                  </>
                );
              },
            },
          ]}
        />
      )}
    </>
  );
};

export default TeamAgentList;
