import React from 'react';
import { useNavigate } from 'react-router-dom';
import { Card } from '@/components/Elements/Card';
import { Button } from '@/components/Elements/Button';
import { AdminLayout } from '@/components/Layout';
import './ContentManagement.css';
import { useContent } from '@/features/admin/api/getContent';
import { postContent } from '@/features/admin/api/postContent';
import * as z from 'zod';
import { Form, InputField, TextAreaField } from '@/components/Form';
import { InvestorContent } from '@/types/InvestorContent';
import ContentManagement from '@/components/Elements/Loader/ContentManagement';
import i18next from 'i18next';

const schema = z.object({
  page_title: z.string().min(1, 'Please enter company name'),
  page_content: z.string().min(1, 'Please enter company established'),
  page_title_nw: z.string().min(1, 'Please enter company name'),
  page_content_nw: z.string().min(1, 'Please enter company established'),
});

type ContactValues = {
  page_title: string;
  page_content: string;
  page_title_nw: string;
  page_content_nw: string;
};

function EditContent() {
  const searchParams = new URLSearchParams(document.location.search);
  const pageName = searchParams.get('pageName');
  const { data, isLoading, isRefetching, refetch } = useContent();
  const user = data?.data ?? ({} as InvestorContent);

  const handleSubmit = async (values: ContactValues) => {
    try {
      await postContent({ ...values });
      refetch();
    } finally {
      refetch();
    }
  };
  const navigate = useNavigate();

  return (
    <>
      <AdminLayout title="Edit Content Management">
        <Card shadow>
          <div className="p-3 pb-0">
            <h5>
              <strong> {pageName} </strong>
            </h5>
          </div>
          {isLoading || isRefetching ? (
            <ContentManagement />
          ) : (
            <Form<ContactValues, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: user,
              }}
            >
              {({ register, formState }) => (
                <>
                  <div className="mx-4">
                    <InputField
                      type="text"
                      label={i18next.t('Title')}
                      error={formState.errors['page_title']}
                      registration={register('page_title')}
                    />
                    <TextAreaField
                      label="Content"
                      rows={4}
                      error={formState.errors['page_content']}
                      registration={register('page_content')}
                    />

                    <InputField
                      type="text"
                      label={i18next.t('Title in NOK')}
                      error={formState.errors['page_title_nw']}
                      registration={register('page_title_nw')}
                    />
                    <TextAreaField
                      label="Content in NOK"
                      rows={4}
                      error={formState.errors['page_content_nw']}
                      registration={register('page_content_nw')}
                    />

                    <div className="d-flex justify-content-end my-5">
                      <Button onClick={() => navigate(-1)} variant="inverse" block className="me-3">
                        {i18next.t(`Cancel`)}
                      </Button>
                      <Button type="submit">Submit</Button>
                    </div>
                  </div>
                </>
              )}
            </Form>
          )}
        </Card>
      </AdminLayout>
    </>
  );
}

export default EditContent;
