import React, { useEffect, useState } from 'react';
import { Button, Table, TableColumn } from '@/components/Elements';
import { useNavigate } from 'react-router-dom';
import { useFaq } from '@/features/admin/api/getFaq';
import { deleteFaq } from '@/features/admin/api/deleteFaq';
import { FaqContent } from '@/types';
import './ContentManagement.css';
import { useNotificationStore } from '@/stores/notifications';

import 'react-responsive-modal/styles.css';
import { Modal } from 'react-responsive-modal';
import TableLoader from '@/components/Elements/Loader/TableLoader';

type Content = {
  id: string;
  title: string;
};

const FaqListing = ({ page }: { page: string }) => {
  const [open, setOpen] = useState(false);
  const onOpenModal = () => setOpen(true);
  const onCloseModal = () => setOpen(false);
  const navigate = useNavigate();
  const { data, refetch, isLoading } = useFaq({ page });
  const [datId, setDataId] = useState('');
  const dataLength = data?.data.length;

  const deleteSelected = async () => {
    await deleteFaq(datId);
    refetch();
    useNotificationStore.getState().addNotification({
      title: 'Success',
      type: 'success',
      message: 'Faq deleted successfully !',
    });
    onCloseModal();
  };
  useEffect(() => {
    refetch();
  }, [page, refetch]);
  const columns: TableColumn<Content>[] = [
    {
      title: 'FAQ Title',
      field: 'title',
      Cell({ entry: { title } }) {
        return <span>{title}</span>;
      },
    },
    {
      title: 'Action',
      field: 'id',
      Cell({ entry: { id } }) {
        return (
          <>
            <span
              role="button"
              tabIndex={-1}
              onKeyDown={() => navigate(`/admin/faq-content-edit/${id}`)}
              onClick={() => navigate(`/admin/faq-content-edit/${id}`)}
              className="icon me-2 buttons2"
              title="Edit"
            >
              <i className="fa-solid fa-pencil"></i>
            </span>
            <span
              role="button"
              tabIndex={-1}
              onKeyDown={() => setDataId(`${id}`)}
              onClick={() => {
                setDataId(`${id}`);
                onOpenModal();
              }}
              className="icon me-2 buttons2"
              title="Edit"
            >
              <i className="fa-solid fa-trash-can deleteTrash"></i>
            </span>
          </>
        );
      },
    },
  ];
  return (
    <>
      {isLoading ? (
        <TableLoader />
      ) : (
        <div>
          <Button
            className="addFaqs"
            onClick={() => {
              navigate('/admin/add-faq-content');
            }}
          >
            Add Faq +
          </Button>
          {data?.data ? (
            <Table<FaqContent>
              data={data?.data ?? []}
              columns={columns}
              page={1}
              total={dataLength}
              changePage={() => null}
            />
          ) : (
            <TableLoader />
          )}
          <Modal open={open} onClose={onCloseModal} center>
            <section className="model-datas">
              <p>Are you Sure you want to delete the FAQ</p>
            </section>
            <p className="btn-lines">
              <Button onClick={deleteSelected}>OKAY</Button>
            </p>
          </Modal>
        </div>
      )}
    </>
  );
};

export default FaqListing;
