import React, { useState } from 'react';
import { AdminLayout } from '@/components/Layout';
import { z } from 'zod';
import { Button } from '@/components/Elements';
import { Form, InputField } from '@/components/Form';
import './General.css';
import { useNotificationStore } from '@/stores/notifications';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { addCommission } from '@/features/admin/api/Commission/addCommission';
import { useCommission } from '@/features/admin/api/Commission/getCommission';
import EditSUbs from '@/components/Elements/Loader/ProfileLoader';

export const GeneralSetting = () => {
  const { data, isLoading, isRefetching, refetch } = useCommission();
  const user = data?.data;
  const schema = z.object({
    commision_value: z
      .string()
      .min(1, 'Percentage value is required')
      .max(2, 'Enter correct % value less then 100%')
      .regex(/^\d+$/)
      .transform(Number),
  });

  type SubscriptionValue = {
    commision_value: number;
  };

  const [loading, setLoading] = useState(false);

  const handleSubmit = async (values: SubscriptionValue) => {
    try {
      setLoading(true);

      await addCommission({
        ...values,
        commision_value: parseInt(values?.commision_value),
      });
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'info',
        message: 'Commission updated successfully!',
      });
      refetch();
    } finally {
      setLoading(false);
    }
  };

  return (
    <AdminLayout title="Commission Management">
      <>
        <BackdropLoader open={loading} />
        <div className="form-boxes">
          {isLoading || isRefetching ? (
            <EditSUbs />
          ) : (
            <Form<SubscriptionValue, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: { commision_value: user?.commision_value?.toString() },
              }}
            >
              {({ register, formState }) => (
                <>
                  <div className="generalDiv row">
                    <div className="col-md-6 col-6">
                      <InputField
                        type="number"
                        label="Commission Percentage"
                        error={formState.errors['commision_value']}
                        registration={register('commision_value')}
                      />
                    </div>
                    <div className="col-md-6 col-6">
                      <div className="btm-btn-centre">
                        <Button type="submit">Submit</Button>
                      </div>
                    </div>
                  </div>
                </>
              )}
            </Form>
          )}
        </div>
      </>
    </AdminLayout>
  );
};
