import React, { useState } from 'react';
import { AdminLayout } from '@/components/Layout';
import { z } from 'zod';
import { Button } from '@/components/Elements';
import { Form, InputField, TextAreaField } from '@/components/Form';
import './Subscription.css';
import { useNotificationStore } from '@/stores/notifications';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useNavigate } from 'react-router-dom';
import { addSubscription } from '../../api/Subscription/addSubscription';

export type SubscriptionValue = {
  id?: string;
  plan_name: string;
  plan_description: string;
  price: string;
  plan_validity: string;
  total_contract: string;
};

export const AddLawyerSubscription = () => {
  const schema = z.object({
    plan_name: z.string().min(1, 'Please enter subscription name'),
    plan_description: z.string().min(1, 'Please enter description'),
    price: z.string().min(1, 'Please enter price '),
    plan_validity: z.string().min(1, 'Please enter validity date '),
    total_contract: z
      .string()
      .min(1, 'Please enter total contracts')
      .regex(/^\d+$/)
      .transform(Number),
  });

  const [loading, setLoading] = useState(false);

  const navigate = useNavigate();
  const handleSubmit = async (values: SubscriptionValue) => {
    try {
      setLoading(true);
      await addSubscription(values);
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'info',
        message: 'Subscription added successfully!',
      });
      navigate('/agency/subsciption');
    } finally {
      setLoading(false);
    }
  };

  return (
    <AdminLayout title="Add Subscription">
      <>
        <BackdropLoader open={loading} />
        <div className="form-boxes">
          <Form<SubscriptionValue, typeof schema> onSubmit={handleSubmit} schema={schema}>
            {({ register, formState }) => (
              <>
                <div className="row">
                  <div className="col-md-6 col-6">
                    <InputField
                      label="Subscription Name"
                      error={formState.errors['plan_name']}
                      registration={register('plan_name')}
                    />
                  </div>
                  <div className="col-md-6 col-6">
                    <InputField
                      label="Subscription Price &#8364;"
                      error={formState.errors['price']}
                      registration={register('price')}
                    />
                  </div>
                </div>
                <div className="row">
                  <div className="col-md-6 col-6">
                    <InputField
                      type="number"
                      label="Plan Validity days"
                      error={formState.errors['plan_validity']}
                      registration={register('plan_validity')}
                    />
                  </div>
                  <div className="col-md-6 col-6">
                    <InputField
                      type="number"
                      label="Total Contracts"
                      error={formState.errors['total_contract']}
                      registration={register('total_contract')}
                    />
                  </div>
                </div>

                <TextAreaField
                  rows={4}
                  error={formState.errors['plan_description']}
                  registration={register('plan_description')}
                  label="Subscription Description"
                />

                <div className="btm-btn-centre">
                  <Button type="submit">Submit</Button>
                </div>
              </>
            )}
          </Form>
        </div>
      </>
    </AdminLayout>
  );
};
