import React, { useState } from 'react';
import { AdminLayout } from '@/components/Layout';
import { z } from 'zod';
import { Button } from '@/components/Elements';
import { Form, InputField, TextAreaField } from '@/components/Form';
import './Subscription.css';
import { useNotificationStore } from '@/stores/notifications';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useNavigate, useParams } from 'react-router-dom';
import EditSUbs from '@/components/Elements/Loader/ProfileLoader';
import { editLawyerSubscription } from '../../api/Subscription/editLawyerSubscription';
import { useLawyerById } from '../../api/Subscription/getLawyerById';

export type LawyerSubscriptionValue = {
  id?: string;
  plan_name: string;
  plan_description: string;
  price: string;
  plan_validity: string;
  total_contract: string;
};

export const EditLawyerSubscription = () => {
  const { id } = useParams();

  const { data, isLoading, isRefetching, refetch } = useLawyerById({ id: id ?? '1' });
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const user: LawyerSubscriptionValue = data?.data ?? ({} as LawyerSubscriptionValue);
  const schema = z.object({
    plan_name: z.string().min(1, 'Please enter subscription name'),
    plan_description: z.string().min(1, 'Please enter description'),
    price: z.string().min(1, 'Please enter price '),
    plan_validity: z.string().min(1, 'Please enter validity date '),
    total_contract: z
      .string()
      .min(1, 'Please enter total contracts')
      .regex(/^\d+$/)
      .transform(Number),
  });

  const [loading, setLoading] = useState(false);

  const navigate = useNavigate();
  const handleSubmit = async (values: LawyerSubscriptionValue) => {
    try {
      setLoading(true);
      await editLawyerSubscription({
        ...values,
        id,
      });
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'info',
        message: 'Subscription added successfully!',
      });
      refetch();
    } finally {
      setLoading(false);
      navigate('/admin');
    }
  };

  return (
    <AdminLayout title="Edit Subscription">
      <>
        <BackdropLoader open={loading} />
        <div className="form-boxes">
          {isLoading || isRefetching ? (
            <EditSUbs />
          ) : (
            <Form<LawyerSubscriptionValue, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: {
                  ...user,
                  total_contract: user.total_contract.toString(),
                },
              }}
            >
              {({ register, formState }) => (
                <>
                  <div className="row">
                    <div className="col-md-6 col-6">
                      <InputField
                        type="text"
                        label="Subscription Name"
                        error={formState.errors['plan_name']}
                        registration={register('plan_name')}
                      />
                    </div>
                    <div className="col-md-6 col-6">
                      <InputField
                        type="number"
                        label="Subscription Price &#8364;"
                        error={formState.errors['price']}
                        registration={register('price')}
                      />
                    </div>
                  </div>
                  <div className="row">
                    <div className="col-md-6 col-6">
                      <InputField
                        type="number"
                        label="Plan Validity days"
                        error={formState.errors['plan_validity']}
                        registration={register('plan_validity')}
                      />
                    </div>
                    <div className="col-md-6 col-6">
                      <InputField
                        type="number"
                        label="Total Contracts"
                        error={formState.errors['total_contract']}
                        registration={register('total_contract')}
                      />
                    </div>
                  </div>

                  <TextAreaField
                    rows={4}
                    error={formState.errors['plan_description']}
                    registration={register('plan_description')}
                    label="Subscription Description"
                  />

                  <div className="btm-btn-centre">
                    <Button type="submit">Submit</Button>
                  </div>
                </>
              )}
            </Form>
          )}
        </div>
      </>
    </AdminLayout>
  );
};
