import React, { useEffect, useState } from 'react';
import { AdminLayout } from '@/components/Layout';
import { z } from 'zod';
import { Button } from '@/components/Elements';
import { Form, InputField, SelectField, TextAreaField } from '@/components/Form';
import './Subscription.css';
import { useNotificationStore } from '@/stores/notifications';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { editSubscription } from '@/features/admin/api/Subscription/editSubscription';
import { useSubId } from '@/features/admin/api/Subscription/getSubId';
import { useNavigate, useParams } from 'react-router-dom';
import { Subscription } from '@/types';
import EditSUbs from '@/components/Elements/Loader/ProfileLoader';
import tireCategory from '@/constants/tireCategory';

export const EditSubscription = () => {
  const { id } = useParams();

  const { data, isLoading, isRefetching, refetch } = useSubId({ id: id ?? '1' });
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const user = data?.data ?? ({} as Subscription);

  const schema = z.object({
    plan_name: z.string().min(1, 'Please enter subscription name'),
    plan_description: z.string().min(1, 'Please enter description'),
    price: z.string().min(1, 'Please enter price '),
    property_count: z.string().min(1, 'Please enter property count '),
    plan_validity: z.string().min(1, 'Please enter validity date '),
    digital_rental_contract: z.boolean(),
    notifications_stakeholders: z.boolean(),
    visibility_among_ads: z.boolean(),
    visibility_at_top: z.boolean(),
    information_of_local_area: z.boolean(),
    tier_type: z.string().min(1, 'Please select tier type'),
    package_type: z.string().min(1, 'Please enter package type'),
    discount: z.number().optional(),
  });

  type SubscriptionValue = {
    plan_name: string;
    plan_description: string;
    price: string;
    property_count: string;
    plan_validity: string;
    digital_rental_contract: boolean;
    notifications_stakeholders: boolean;
    visibility_among_ads: boolean;
    visibility_at_top: boolean;
    information_of_local_area: boolean;
    tier_type: any;
    package_type: string;
    discount: number;
  };

  const [digital_rental_contract, setcheck1] = useState<boolean>(false);
  const [notifications_stakeholders, setcheck2] = useState<boolean>(false);
  const [visibility_among_ads, setcheck3] = useState<boolean>(false);
  const [visibility_at_top, setcheck4] = useState<boolean>(false);
  const [information_of_local_area, setcheck5] = useState<boolean>(false);
  const [discountClick, setDiscount] = useState<boolean>(false);

  useEffect(() => {
    if (user) {
      setcheck1(!!user.digital_rental_contract);
      setcheck2(!!user.notifications_stakeholders);
      setcheck3(!!user.visibility_among_ads);
      setcheck4(!!user.visibility_at_top);
      setcheck5(!!user.information_of_local_area);
    }
  }, [user]);

  const [loading, setLoading] = useState(false);

  const handleCheck1 = () => {
    setcheck1((current) => !current);
  };

  const handleCheck2 = () => {
    setcheck2((current) => !current);
  };

  const handleCheck3 = () => {
    setcheck3((current) => !current);
  };

  const handleCheck4 = () => {
    setcheck4((current) => !current);
  };

  const handleCheck5 = () => {
    setcheck5((current) => !current);
  };

  const navigate = useNavigate();
  const handleSubmit = async (values: SubscriptionValue) => {
    const x = values.tier_type.toString();
    try {
      setLoading(true);

      await editSubscription({
        ...values,
        discount: discountClick ? values.discount : 0,
        tier_type: x,
        digital_rental_contract,
        notifications_stakeholders,
        visibility_among_ads,
        visibility_at_top,
        information_of_local_area,
        id,
      });
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'info',
        message: 'Subscription added successfully!',
      });
      refetch();
    } finally {
      setLoading(false);
      navigate('/admin/subscription-management');
    }
  };

  const packageOptions = [
    {
      label: 'Monthly',
      value: 'monthly',
    },
    {
      label: 'Yearly',
      value: 'yearly',
    },
  ];

  const handlePackageType = (val: string) => {
    if (val === 'yearly') {
      setDiscount(true);
    } else {
      setDiscount(false);
    }
  };

  useEffect(() => {
    if (user?.package_type === 'yearly') {
      setDiscount(true);
    }
  }, [isLoading, isRefetching]);

  return (
    <AdminLayout title="Edit Subscription">
      <>
        <BackdropLoader open={loading} />
        <div className="form-boxes">
          {isLoading || isRefetching ? (
            <EditSUbs />
          ) : (
            <Form<SubscriptionValue, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: { ...user, tier_type: user?.tier_type.toString() },
              }}
            >
              {({ register, formState }) => (
                <>
                  <div className="row">
                    <div className="col-md-6 col-6">
                      <InputField
                        type="text"
                        label="Subscription Name"
                        error={formState.errors['plan_name']}
                        registration={register('plan_name')}
                      />
                    </div>
                    <div className="col-md-6 col-6">
                      <InputField
                        type="number"
                        label="Subscription Price &#8364;"
                        error={formState.errors['price']}
                        registration={register('price')}
                      />
                    </div>
                  </div>
                  <div className="row">
                    <div className="col-md-6 col-6">
                      <InputField
                        type="number"
                        label="Plan Validity"
                        error={formState.errors['plan_validity']}
                        registration={register('plan_validity')}
                      />
                    </div>
                    <div className="col-md-6 col-6">
                      <InputField
                        label="Property Count"
                        error={formState.errors['property_count']}
                        registration={register('property_count')}
                      />
                    </div>
                  </div>

                  <div className="row">
                    <div className="col-md-6 col-6">
                      <SelectField
                        wrapperClass="w-full"
                        label="Tier Type"
                        error={formState.errors['tier_type']}
                        registration={register('tier_type')}
                        placeholder="Tier Type"
                        options={Object.entries(tireCategory).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                      />
                    </div>
                    <div className="col-md-6 col-6">
                      <SelectField
                        wrapperClass="w-full"
                        label="Package Type"
                        error={formState.errors['package_type']}
                        registration={register('package_type', {
                          onChange: (e) => handlePackageType(e.target.value),
                        })}
                        placeholder="Package Type"
                        options={packageOptions}
                      />
                    </div>
                    {discountClick && (
                      <div className="col-md-6 col-6">
                        <InputField
                          type="number"
                          label="Discount in percentage %"
                          error={formState.errors['discount']}
                          registration={register('discount', { valueAsNumber: true })}
                        />
                      </div>
                    )}
                  </div>
                  <TextAreaField
                    rows={4}
                    error={formState.errors['plan_description']}
                    registration={register('plan_description')}
                    label="Subscription Description"
                  />

                  <div className="row">
                    <div className="subs-checkboxes">
                      <label htmlFor="digital_rental_contract">
                        <input
                          type="checkbox"
                          onChange={handleCheck1}
                          id="digital_rental_contract"
                          name="digital_rental_contract"
                          checked={digital_rental_contract}
                        />
                        <span className="subs-labl">Digital Rental Contract</span>
                      </label>
                    </div>

                    <div className="subs-checkboxes">
                      <label htmlFor="notifications_stakeholders">
                        <input
                          type="checkbox"
                          onChange={handleCheck2}
                          id="notifications_stakeholders"
                          name="notifications_stakeholders"
                          checked={notifications_stakeholders}
                        />
                        <span className="subs-labl">Notifications Stakeholders</span>
                      </label>
                    </div>

                    <div className="subs-checkboxes">
                      <label htmlFor="visibility_among_ads">
                        <input
                          type="checkbox"
                          onChange={handleCheck3}
                          id="visibility_among_ads"
                          name="visibility_among_ads"
                          checked={visibility_among_ads}
                        />
                        <span className="subs-labl">Visibility Among Ads</span>
                      </label>
                    </div>

                    <div className="subs-checkboxes">
                      <label htmlFor="visibility_at_top">
                        <input
                          type="checkbox"
                          onChange={handleCheck4}
                          id="visibility_at_top"
                          name="visibility_at_top"
                          checked={visibility_at_top}
                        />
                        <span className="subs-labl">Visibility At Top</span>
                      </label>
                    </div>

                    <div className="subs-checkboxes">
                      <label htmlFor="information_of_local_area">
                        <input
                          type="checkbox"
                          onChange={handleCheck5}
                          id="information_of_local_area"
                          name="information_of_local_area"
                          checked={information_of_local_area}
                        />
                        <span className="subs-labl">Information Of Local Area</span>
                      </label>
                    </div>
                  </div>
                  <div className="btm-btn-centre">
                    <Button type="submit">Submit</Button>
                  </div>
                </>
              )}
            </Form>
          )}
        </div>
      </>
    </AdminLayout>
  );
};
