import React, { useRef, useState } from 'react';
import * as z from 'zod';
import { useNavigate } from 'react-router-dom';
import { ContentLayout } from '@/components/Layout';
import { Form, InputField, TextAreaField } from '@/components/Form';

import ProfileLoader from '@/components/Elements/Loader/ProfileLoader';
import { Button, Image } from '@/components/Elements';
import { useSelectFile } from '@/hooks/useSelectFile';
import { uploadImage } from '@/api/uploadImage';
import { useNotificationStore } from '@/stores/notifications';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useAuth } from '@/lib/auth';
import { InputPhone } from '@/components/Form/InputPhone';
import { isValidMobile, isValidMobileMsg } from '@/utils/validation';
import GooglePlacesAutocomplete from 'react-google-places-autocomplete';
import axiosmod from 'axios';
import { updateAgencyProfile } from '@/features/auth/api/updateAgencyProfile';

import { AgencyData, useAgencyProfileData } from '@/features/auth/api/getAgecyProfile';
import i18next from 'i18next';

const googleMapAPIkey = process.env.REACT_APP_GOOGLE_MAPS_API_KEY as string;

const schema = z.object({
  first_name: z.string().min(1, 'Please enter first name'),
  last_name: z.string().min(1, 'Please enter last name'),
  email: z.string().optional(),
  phone_number: z.custom(isValidMobile, isValidMobileMsg),
  your_self: z.string().min(1, 'Please enter description'),
  company: z.string().optional(),
});

type UpdateProfileValues = {
  first_name: string;
  last_name: string;
  email: string;
  address: string;
  phone_number: string;
  your_self: string;
  company: string;
};

type Coordinates = {
  lat: number;
  lng: number;
};

export const AgencyProfile = () => {
  const { refetchUser } = useAuth();
  const navigate = useNavigate();
  const inputFileRef = useRef<HTMLInputElement | null>(null);
  const { file, preview: picPreview, handleChange } = useSelectFile();
  const [loading, setLoading] = useState(false);

  const [address, setAddress] = useState<string>('');
  const [locationGoogle, setLocationGoogle] = useState<Coordinates>();

  const [value, setValue] = useState(null);

  const { data, isRefetching, isLoading, refetch } = useAgencyProfileData();

  // ------------------ Location ---------------------

  const handleChangeLocation = async (value: any) => {
    const placeId = value.value.place_id;
    const { data } = await axiosmod.get(
      `https://maps.googleapis.com/maps/api/geocode/json?place_id=${placeId}&key=${googleMapAPIkey}`
    );
    const location = data?.results[0]?.geometry?.location;
    const addressData = data?.results[0]?.formatted_address;
    console.log({ data, location, addressData });

    if (location) {
      setLocationGoogle(location);
      setValue(value);
    }
    if (addressData) {
      setAddress(addressData);
    }
  };

  const profile = data?.data ?? ({} as AgencyData);
  console.log(profile);
  const handleSubmit = async (values: UpdateProfileValues) => {
    try {
      setLoading(true);
      let profile_image = profile?.profile_image ?? '';

      if (file) {
        const response = await uploadImage({ file });
        profile_image = response.url;
      }
      await updateAgencyProfile({
        ...values,
        address: address ? address : profile?.address,
        profile_image,
        latitude: locationGoogle?.lat ? locationGoogle?.lat : profile?.latitude,
        longitude: locationGoogle?.lng ? locationGoogle?.lng : profile?.longitude,
      });
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'Profile updated successfully!',
      });
      refetchUser();
      refetch();
      navigate('/agency');
    } finally {
      setLoading(false);
    }
  };

  const changeClicked = () => {
    inputFileRef?.current?.click();
  };

  return (
    <ContentLayout title={i18next.t('Profile')}>
      <BackdropLoader open={loading} />
      <div className="container my-4">
        <div className=" ">
          {isLoading || isRefetching ? (
            <ProfileLoader />
          ) : (
            <Form<UpdateProfileValues, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: {
                  ...profile,
                  company: profile?.agency_company,
                },
              }}
            >
              {({ register, formState }) => (
                <>
                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Profile Picture</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      {picPreview ? (
                        <img alt="banner" src={picPreview} className="previewImage" />
                      ) : (
                        <Image src={profile?.profile_image} alt="" className="previewImage" />
                      )}
                      <Button onClick={changeClicked} variant="outline" size="sm" className="mt-2">
                        Change
                      </Button>
                      <input
                        ref={inputFileRef}
                        onChange={handleChange}
                        type="file"
                        id="fileInput"
                        className="d-none"
                      />
                    </div>
                  </div>

                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>First Name</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputField
                        type="text"
                        label="First Name"
                        error={formState.errors['first_name']}
                        registration={register('first_name')}
                        placeholder="First Name"
                      />
                    </div>
                  </div>
                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Last Name</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputField
                        type="text"
                        label="Last Name"
                        error={formState.errors['last_name']}
                        registration={register('last_name')}
                        placeholder="Last Name"
                      />
                    </div>
                  </div>
                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>{i18next.t('Email Address')}</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputField
                        type="email"
                        label={i18next.t('Email Address')}
                        disabled
                        error={formState.errors['email']}
                        registration={register('email')}
                        placeholder={i18next.t('Email Address')}
                      />
                    </div>
                  </div>

                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Telephone</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputPhone
                        floating
                        value={profile?.phone_number}
                        error={formState.errors['phone_number']}
                        registration={register('phone_number')}
                      />
                    </div>
                  </div>

                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Address</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <GooglePlacesAutocomplete
                        apiKey={googleMapAPIkey}
                        selectProps={{
                          defaultInputValue: `${profile?.address}`,
                          value,
                          isClearable: true,
                          placeholder: `${i18next.t('Search for city/location')}`,
                          onChange: handleChangeLocation,
                        }}
                      />
                    </div>
                  </div>

                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Company Detaildds</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputField
                        type="text"
                        label={i18next.t('Company Name')}
                        error={formState.errors['company']}
                        registration={register('company')}
                        placeholder={i18next.t('Company Name')}
                      />
                    </div>
                  </div>

                  <div className="row align-items-start">
                    <div className="col-12 col-md-3">
                      <h5>Tell About Yourself</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <TextAreaField
                        rows={5}
                        label="Tell About Your Self"
                        error={formState.errors['your_self']}
                        registration={register('your_self')}
                      />
                    </div>
                  </div>

                  <div>
                    <Button isLoading={loading} type="submit" className="mx-auto d-inline-block">
                      Update Account
                    </Button>
                  </div>
                </>
              )}
            </Form>
          )}
        </div>
      </div>
    </ContentLayout>
  );
};
