import React, { useEffect, useState } from 'react';
import { Button, Table } from '@/components/Elements';
import { Inquiry } from '@/types';
import TableLoader from '@/components/Elements/Loader/TableLoader';
import { ConfirmDialogWithButton } from '@/components/Elements/ConfirmationDialog/ConfirmDialogWithButton';
import { changeStatus } from '@/api/inquiry/changeStatus';
import { useNotificationStore } from '@/stores/notifications';
import { useNavigate } from 'react-router-dom';
import { useInquiry } from '@/api/inquiry/getInquiry';
import i18next from 'i18next';

const InquiryList = ({
  page,
  changePage,
}: {
  page: string;
  changePage: (page: string) => void;
}) => {
  const { data, refetch, isLoading, isFetching } = useInquiry({ page });
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();
  const [moddedInquiry, setModdedInquiry] = useState<any>([]);

  useEffect(() => {
    let inquiriesMod: any = [];

    inquiriesMod = data?.data.map((item) => {
      const inquiry: any = {};
      inquiry.id = item.id;
      inquiry.name = item.name;
      inquiry.telephone = item.telephone;
      inquiry.email = item.email;
      inquiry.message = item.message;
      inquiry.type = item.type;
      inquiry.status = item.status == 1 ? 'Approved' : item.status == 0 ? 'Pending' : 'Rejected';

      return inquiry;
    });
    setModdedInquiry(inquiriesMod);
  }, [data]);

  useEffect(() => {
    refetch();
  }, [page, refetch]);

  const changeStatusAction = async (id: string, accept: boolean) => {
    try {
      setLoading(true);
      await changeStatus(id, accept);
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'You have approved this inquiry successfully!',
      });
      refetch();
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
      {isLoading || isFetching ? (
        <TableLoader />
      ) : (
        <Table<Inquiry>
          page={parseInt(page)}
          changePage={changePage}
          total={data?.total ?? 0}
          data={moddedInquiry ?? []}
          columns={[
            { title: `${i18next.t('Name')}`, field: 'name' },
            { title: 'Telephone', field: 'telephone' },
            { title: `${i18next.t('Email')}`, field: 'email' },
            // { title: 'Message', field: 'message' },
            { title: 'Status', field: 'status' },
            {
              title: '',
              field: 'status',
              Cell({ entry: { status, id } }) {
                return (
                  <>
                    {status === 'Pending' && (
                      <ConfirmDialogWithButton
                        triggerButton={<Button variant="outlineDark">Approve</Button>}
                        confirmButton={
                          <Button
                            isLoading={loading}
                            onClick={() => changeStatusAction(id, true)}
                            variant="primary"
                          >
                            Approve
                          </Button>
                        }
                        title="Confirm"
                        icon="info"
                        body="Are you sure you want to approve this inquiry?"
                      />
                    )}
                  </>
                );
              },
            },
            {
              title: 'Action',
              field: 'status',
              Cell({ entry: { id, type } }) {
                return (
                  <>
                    <span
                      role="button"
                      tabIndex={-1}
                      onKeyDown={() => navigate(`/admin/inquiry/${id}&type=${type}`)}
                      onClick={() => navigate(`/admin/inquiry/${id}&type=${type}`)}
                      className="icon me-2"
                      title="View"
                    >
                      <i className="fa-regular fa-eye" />
                    </span>
                  </>
                );
              },
            },
          ]}
        />
      )}
    </>
  );
};

export default InquiryList;
