import React, { useState } from 'react';
import { AdminLayout } from '@/components/Layout';
import { z } from 'zod';
import { Button } from '@/components/Elements';
import { Form, InputField, SelectField, TextAreaField } from '@/components/Form';
import './Subscription.css';
import { useNotificationStore } from '@/stores/notifications';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useNavigate } from 'react-router-dom';
import tireCategory from '@/constants/tireCategory';
import { addSubScription } from '../../api/addSubScription';

export type SubscriptionValue = {
  id?: string;
  plan_name: string;
  plan_description: string;
  price: string;
  plan_validity: string;
  tier_type: string;
};

export const AddSubscriptionAgency = () => {
  const schema = z.object({
    plan_name: z.string().min(1, 'Please enter subscription name'),
    plan_description: z.string().min(1, 'Please enter description'),
    price: z.string().min(1, 'Please enter price '),
    plan_validity: z.string().min(1, 'Please enter validity date '),
    tier_type: z.string().min(1, 'Please select tier type'),
  });

  const [loading, setLoading] = useState(false);

  const navigate = useNavigate();
  const handleSubmit = async (values: SubscriptionValue) => {
    try {
      setLoading(true);
      await addSubScription(values);
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'info',
        message: 'Subscription added successfully!',
      });
    } finally {
      setLoading(false);
      navigate('/agency/subsciption');
    }
  };

  return (
    <AdminLayout title="Add Subscription">
      <>
        <BackdropLoader open={loading} />
        <div className="form-boxes">
          <Form<SubscriptionValue, typeof schema> onSubmit={handleSubmit} schema={schema}>
            {({ register, formState }) => (
              <>
                <div className="row">
                  <div className="col-md-6 col-6">
                    <InputField
                      label="Subscription Name"
                      error={formState.errors['plan_name']}
                      registration={register('plan_name')}
                    />
                  </div>
                  <div className="col-md-6 col-6">
                    <InputField
                      label="Subscription Price &#8364;"
                      error={formState.errors['price']}
                      registration={register('price')}
                    />
                  </div>
                </div>
                <div className="row">
                  <div className="col-md-6 col-6">
                    <InputField
                      type="number"
                      label="Plan Validity days"
                      error={formState.errors['plan_validity']}
                      registration={register('plan_validity')}
                    />
                  </div>
                  <div className="col-md-6 col-6">
                    <SelectField
                      wrapperClass="w-full"
                      label="Tier Type"
                      error={formState.errors['tier_type']}
                      registration={register('tier_type')}
                      placeholder="Tier Type"
                      options={Object.entries(tireCategory).map(([key, value]) => ({
                        label: value,
                        value: key,
                      }))}
                    />
                  </div>
                </div>
                <TextAreaField
                  rows={4}
                  error={formState.errors['plan_description']}
                  registration={register('plan_description')}
                  label="Subscription Description"
                />

                <div className="btm-btn-centre">
                  <Button type="submit">Submit</Button>
                </div>
              </>
            )}
          </Form>
        </div>
      </>
    </AdminLayout>
  );
};
