import React, { useState } from 'react';
import * as z from 'zod';

import { ContentLayout } from '@/components/Layout';
import { Form, InputField, SelectField, TextAreaField } from '@/components/Form';
import companyCategory from '@/constants/companyCategory';

import { Button } from '@/components/Elements';
import { useNotificationStore } from '@/stores/notifications';
import { useNavigate } from 'react-router-dom';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import '../styles/style.css';
import { createCompany } from '@/api/company/createCompany';
import { useCompany } from '@/api/company/updateCompany';
import { Company } from '@/types/Company';
import CompanyLoader from '@/components/Elements/Loader/CompanyLoader';
import i18next from 'i18next';

const schema = z.object({
  company_name: z.string().min(1, 'Please enter company name'),
  company_url: z.string().min(1, 'Please enter company url'),
  company_email: z
    .string()
    .min(1, 'Please enter company email address')
    .email('Please enter a valid email address'),
  company_telephone: z
    .string()
    .min(1, 'Please enter company phone')
    .regex(/^[-0-9+]+$/, 'Phone should contain numbers only!'),
  company_slogan: z.string().min(1, 'Please enter company slogan'),
  company_category: z.string().min(1, 'Please enter company category'),
  facebookLink: z.string().optional(),
  twitterLink: z.string().optional(),
  youtubeLink: z.string().optional(),
  linkedinLink: z.string().optional(),
  company_detail: z.string().optional(),
  companyVision: z.string().optional(),
});

type CreateCompanyValues = {
  company_name: string;
  company_url: string;
  company_slogan: string;
  company_email: string;
  companyVision: string;
  company_telephone: string;
  company_detail: string;
  twitterLink: string;
  facebookLink: string;
  youtubeLink: string;
  linkedinLink: string;
  company_category: string;
};

export const AddCompany = () => {
  const { data, isLoading, isRefetching, refetch } = useCompany();
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);

  const company = data?.data ?? ({} as Company);

  const handleSubmit = async (values: CreateCompanyValues) => {
    try {
      setLoading(true);
      await createCompany(values);
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'Company added successfully!',
      });
      refetch();
      navigate('/');
    } finally {
      setLoading(false);
    }
  };
  return (
    <ContentLayout title="Add Company">
      <BackdropLoader open={loading} />
      <button onClick={() => navigate(-1)}>
        <i className="fa-regular fa-circle-left go-back-button"></i>
      </button>
      <div className="container mt-4">
        <h3>Company Profile</h3>

        {isLoading || isRefetching ? (
          <CompanyLoader />
        ) : (
          <Form<CreateCompanyValues, typeof schema>
            onSubmit={handleSubmit}
            schema={schema}
            options={{
              defaultValues: {
                company_name: company.companyName,
                company_url: company.companyUrl,
                company_slogan: company.companySlogan,
                company_email: company.companyEmail,
                companyVision: company.companyVision,
                company_telephone: company?.companyTelephone?.toString(),
                company_detail: company.companyDetail,
                twitterLink: company.twitterLink,
                facebookLink: company.facebookLink,
                youtubeLink: company.youtubeLink,
                linkedinLink: company.linkedinLink,
                company_category: company.companyCategory,
              },
            }}
          >
            {({ register, formState }) => (
              <div className="profile-box mb-5">
                <div className="row mt-4">
                  <div className="formDiv">
                    <p>{i18next.t('Company name')}</p>
                    <InputField
                      type="text"
                      error={formState.errors['company_name']}
                      registration={register('company_name')}
                      placeholder={i18next.t('Company name')}
                    />
                  </div>
                  <div className="formDiv">
                    <p>Company Slug / Url</p>
                    <InputField
                      type="text"
                      error={formState.errors['company_url']}
                      registration={register('company_url')}
                      placeholder="Company Slug / URL"
                    />
                  </div>
                  <div className="formDiv">
                    <p>Slogan</p>
                    <InputField
                      type="text"
                      min={0}
                      error={formState.errors['company_slogan']}
                      registration={register('company_slogan')}
                      placeholder="Company Slogan"
                    />
                  </div>
                  <div className="formDiv">
                    <p>Company Category</p>
                    <SelectField
                      error={formState.errors['company_category']}
                      registration={register('company_category')}
                      placeholder="Company Category"
                      options={Object.entries(companyCategory).map(([key, value]) => ({
                        label: value,
                        value: key,
                      }))}
                    />
                  </div>
                  <div className="formDiv">
                    <p>Company Email Address</p>
                    <InputField
                      type="email"
                      error={formState.errors['company_email']}
                      registration={register('company_email')}
                      placeholder="Company Email Address"
                    />
                  </div>
                  <div className="formDiv">
                    <p>Company Telephone</p>
                    <InputField
                      type="text"
                      error={formState.errors['company_telephone']}
                      registration={register('company_telephone')}
                      placeholder="Company Telephone"
                    />
                  </div>
                  <div className="formDiv">
                    <p>Company Vision</p>
                    <TextAreaField
                      rows={4}
                      error={formState.errors['companyVision']}
                      registration={register('companyVision')}
                    />
                  </div>
                  <div className="formDiv">
                    <p>{i18next.t('Company Details')}</p>
                    <TextAreaField
                      rows={4}
                      error={formState.errors['company_detail']}
                      registration={register('company_detail')}
                    />
                  </div>
                  <div className="formDiv">
                    <p>Company Social Media</p>
                    <div>
                      <div className="col-12">
                        <InputField
                          type="text"
                          label="Facebook"
                          error={formState.errors['facebookLink']}
                          registration={register('facebookLink')}
                          placeholder="Facebook"
                        />
                      </div>
                      <div className="col-12">
                        <InputField
                          type="text"
                          label="Twitter"
                          error={formState.errors['twitterLink']}
                          registration={register('twitterLink')}
                          placeholder="Twitter"
                        />
                      </div>
                      <div className="col-12 mt-2">
                        <InputField
                          type="text"
                          label={i18next.t('Youtube')}
                          error={formState.errors['youtubeLink']}
                          registration={register('youtubeLink')}
                          placeholder={i18next.t('Youtube')}
                        />
                      </div>
                      <div className="col-12 mt-2">
                        <InputField
                          type="text"
                          label="Linkedin"
                          error={formState.errors['linkedinLink']}
                          registration={register('linkedinLink')}
                          placeholder="Linkedin"
                        />
                      </div>
                    </div>
                  </div>
                  <div className="d-flex justify-content-end mt-4">
                    <Button isLoading={loading} type="submit" className="mx-auto d-inline-block">
                      Save Changes
                    </Button>
                  </div>
                </div>
              </div>
            )}
          </Form>
        )}
      </div>
    </ContentLayout>
  );
};
