import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { Form, InputField } from '@/components/Form';
import { ContentLayout } from '@/components/Layout';
import { useAuth } from '@/lib/auth';
import React, { useRef, useState } from 'react';
import { useAdminProfile } from '../../api/getAdminProfile';
import * as z from 'zod';
import { Button } from '@/components/Elements/Button';
import { useNavigate } from 'react-router-dom';
import { axios } from '@/lib/axios';
import { useSelectFile } from '@/hooks/useSelectFile';
import AdminProfileLoader from '@/components/Elements/Loader/AdminProfileLoader';

import { InputPhone } from '@/components/Form/InputPhone';
import { isValidMobile, isValidMobileMsg } from '@/utils/validation';
import i18next from 'i18next';
const schema = z.object({
  first_name: z.string().min(1, 'Please enter first name'),
  last_name: z.string().min(1, 'Please enter last name'),
  email: z.string().optional(),
  phone_number: z.custom(isValidMobile, isValidMobileMsg),
});

type UpdateProfileValues = {
  email: string;
  phone_number: string;
  first_name: string;
  last_name: string;
};

type Admin = {
  email: string;
  phone_number: string;
  avatar: string;
  first_name: string;
  last_name: string;
};

const AdminProfile = () => {
  const navigate = useNavigate();
  const { data, isLoading, isRefetching, refetch } = useAdminProfile();
  const { refetchUser } = useAuth();
  const [loading, setLoading] = useState(false);
  const inputFileRef = useRef<HTMLInputElement | null>(null);
  const profile = data?.data ?? ({} as Admin);
  const { file, preview: picPreview, handleChange } = useSelectFile();

  const handleSubmit = async (values: UpdateProfileValues) => {
    const form = new FormData();
    form.append('first_name', values.first_name);
    form.append('last_name', values.last_name);
    form.append('phone_number', values.phone_number);
    form.append('email', values.email);

    if (file) {
      const extension = file?.name.split('.').at(-1);
      const imgname = new Date().getTime() + 'image.' + extension;
      form.append('avatar', file, imgname);
    }
    setLoading(true);
    axios
      .post('admin-api/admin-profile', form)
      .then(() => {
        setLoading(false);
        refetchUser();
        refetch();
      })
      .catch(() => {
        setLoading(false);
      });
  };

  const changeClicked = () => {
    inputFileRef?.current?.click();
  };

  return (
    <ContentLayout title="Admin Profile">
      <BackdropLoader open={loading} />

      <div className="container my-4">
        <div className="profile-box ">
          <h4>Admin Profile</h4>
          {isLoading || isRefetching ? (
            <AdminProfileLoader />
          ) : (
            <Form<UpdateProfileValues, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: profile,
              }}
            >
              {({ register, formState }) => (
                <>
                  <div className="row align-items-center mt-4">
                    <div className="col-8" style={{ marginBottom: '60px' }}>
                      <div className="flex gap-4 items-center">
                        {picPreview ? (
                          <img alt="banner" src={picPreview} className="previewImage" />
                        ) : (
                          <img
                            src={profile?.avatar ?? picPreview}
                            alt=""
                            className="previewImage"
                          />
                        )}
                        <div>
                          <h5>
                            {profile?.first_name} {profile?.last_name}
                          </h5>
                          <Button
                            onClick={changeClicked}
                            variant="primary"
                            size="sm"
                            className="mt-2"
                          >
                            Change
                          </Button>
                        </div>
                      </div>

                      <input
                        ref={inputFileRef}
                        onChange={handleChange}
                        type="file"
                        id="fileInput"
                        className="d-none"
                      />
                    </div>
                    <div className="col-6">
                      <h5>First Name</h5>
                      <InputField
                        type="text"
                        label="First Name"
                        error={formState.errors['first_name']}
                        registration={register('first_name')}
                        placeholder="John"
                      />
                    </div>
                    <div className="col-6">
                      <h5>Last Name</h5>
                      <InputField
                        type="text"
                        label="Last Name"
                        error={formState.errors['last_name']}
                        registration={register('last_name')}
                        placeholder="Doe"
                      />
                    </div>
                    <div className="col-6">
                      <h5>{i18next.t('Phone')}</h5>
                      <InputPhone
                        floating
                        value={profile?.phone_number}
                        error={formState.errors['phone_number']}
                        registration={register('phone_number')}
                      />
                    </div>
                    <div className="col-6">
                      <h5>{i18next.t('Email')}</h5>
                      <InputField
                        type="text"
                        label={i18next.t('Email')}
                        disabled
                        error={formState.errors['email']}
                        registration={register('email')}
                        placeholder="jdoe@gmail.com"
                      />
                    </div>
                  </div>
                  <div className="flex justify-center">
                    <div className="flex gap-2">
                      <Button
                        isLoading={loading}
                        variant="outline"
                        size="sm"
                        className="mt-2"
                        onClick={() => navigate(-1)}
                      >
                        {i18next.t(`Cancel`)}
                      </Button>
                      <Button
                        isLoading={loading}
                        type="submit"
                        variant="primary"
                        size="sm"
                        className="mt-2"
                      >
                        Update
                      </Button>
                    </div>
                  </div>
                </>
              )}
            </Form>
          )}
        </div>
      </div>
    </ContentLayout>
  );
};

export default AdminProfile;
