import React, { useState } from 'react';
import { AdminLayout } from '@/components/Layout';
import { z } from 'zod';
import { Button } from '@/components/Elements';
import { Form, InputField, SelectField, TextAreaField } from '@/components/Form';
import './Subscription.css';
import { useNotificationStore } from '@/stores/notifications';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useNavigate, useParams } from 'react-router-dom';
import { Subscription } from '@/types';
import EditSUbs from '@/components/Elements/Loader/ProfileLoader';
import tireCategory from '@/constants/tireCategory';
import { useAgencySubId } from '../../api/Subscription/getAgencySubId';
import { editAgencySubScription } from '../../api/Subscription/editSubscription';

export const EditSubscriptionLawyer = () => {
  const { id } = useParams();

  const { data, isLoading, isRefetching, refetch } = useAgencySubId({ id: id ?? '1' });
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const user = data?.data ?? ({} as Subscription);
  const schema = z.object({
    plan_name: z.string().min(1, 'Please enter subscription name'),
    plan_description: z.string().min(1, 'Please enter description'),
    price: z.string().min(1, 'Please enter price '),
    plan_validity: z.string().min(1, 'Please enter validity date '),
    tier_type: z.number().min(1, 'Please select tier type'),
  });

  type SubscriptionValue = {
    plan_name: string;
    plan_description: string;
    price: string;
    plan_validity: string;
    tier_type: any;
  };

  const [loading, setLoading] = useState(false);

  const navigate = useNavigate();
  const handleSubmit = async (values: SubscriptionValue) => {
    const x = values.tier_type.toString();
    try {
      setLoading(true);
      await editAgencySubScription({
        ...values,
        tier_type: x,
        id,
      });
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'info',
        message: 'Subscription added successfully!',
      });
      refetch();
    } finally {
      setLoading(false);
      navigate('/agency/subsciption');
    }
  };

  return (
    <AdminLayout title="Edit Subscription">
      <>
        <BackdropLoader open={loading} />
        <div className="form-boxes">
          {isLoading || isRefetching ? (
            <EditSUbs />
          ) : (
            <Form<SubscriptionValue, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: user,
              }}
            >
              {({ register, formState }) => (
                <>
                  <div className="row">
                    <div className="col-md-6 col-6">
                      <InputField
                        type="text"
                        label="Subscription Name"
                        error={formState.errors['plan_name']}
                        registration={register('plan_name')}
                      />
                    </div>
                    <div className="col-md-6 col-6">
                      <InputField
                        type="number"
                        label="Subscription Price &#8364;"
                        error={formState.errors['price']}
                        registration={register('price')}
                      />
                    </div>
                  </div>
                  <div className="row">
                    <div className="col-md-6 col-6">
                      <InputField
                        type="number"
                        label="Plan Validity days"
                        error={formState.errors['plan_validity']}
                        registration={register('plan_validity')}
                      />
                    </div>
                    <div className="col-md-6 col-6">
                      <SelectField
                        wrapperClass="w-full"
                        label="Tier Type"
                        error={formState.errors['tier_type']}
                        registration={register('tier_type')}
                        placeholder="Tier Type"
                        options={Object.entries(tireCategory).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                      />
                    </div>
                  </div>
                  <TextAreaField
                    rows={4}
                    error={formState.errors['plan_description']}
                    registration={register('plan_description')}
                    label="Subscription Description"
                  />

                  <div className="btm-btn-centre">
                    <Button type="submit">Submit</Button>
                  </div>
                </>
              )}
            </Form>
          )}
        </div>
      </>
    </AdminLayout>
  );
};
