import React, { useState } from 'react';
import * as z from 'zod';
import intro2 from '@/assets/intro2.png';
import { useParams } from 'react-router-dom';
import { Form, InputField } from '@/components/Form';
import { Button } from '@/components/Elements';
import { inquiryAd } from '@/api/inquiryAd/inquiryAd';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useAuth } from '@/lib/auth';
import { usePropertyForSale } from '@/api/propertyForSale/getPropertyForSale';
import { InputPhone } from '@/components/Form/InputPhone';
import { isValidMobile, isValidMobileMsg } from '@/utils/validation';
import { translateAmenties } from '@/translations/translateFunction';
import i18next from 'i18next';

export const Inquiry = () => {
  const { id } = useParams();
  const { data } = usePropertyForSale({ id: id ?? '2' });
  const Property = data?.data;
  const customerId = Property?.customer_id;
  const auth = useAuth();
  const loggedinId = auth.user?.customer_id;

  const schema = z.object({
    name: z.string().min(1, 'Please enter your full name'),
    email: z
      .string()
      .min(1, 'Please enter email address')
      .email('Please enter valid email address!'),
    telephone: z.custom(isValidMobile, isValidMobileMsg),

    message: z.string().min(1, 'Please enter a message'),
  });

  type inquiryAd = {
    name: string;
    telephone: string;
    email: string;
    message: string;
  };
  const [loading, setLoading] = useState(false);
  const handleSubmit = async (values: inquiryAd) => {
    try {
      setLoading(true);

      await inquiryAd({
        ...values,
        id: `${id}`,
      });
    } finally {
      setLoading(false);
    }
  };
  return (
    <>
      <BackdropLoader open={loading} />
      <div className="inquiry">
        <h5>Inquire About the Add</h5>
        <div className="row">
          <div className="col-md-7 left">
            <div className="form-box">
              <fieldset disabled={loggedinId === customerId}>
                <Form<inquiryAd> onSubmit={handleSubmit} schema={schema}>
                  {({ register, formState }) => (
                    <>
                      <InputField
                        label={translateAmenties('Name')}
                        error={formState.errors['name']}
                        registration={register('name')}
                      />
                      <InputPhone
                        floating
                        error={formState.errors['telephone']}
                        registration={register('telephone')}
                      />
                      <InputField
                        label={translateAmenties('Email')}
                        error={formState.errors['email']}
                        registration={register('email')}
                      />
                      <InputField
                        label={translateAmenties('Message')}
                        error={formState.errors['message']}
                        registration={register('message')}
                      />
                      {loggedinId === customerId && (
                        <p className="messages">You are owner of this Property</p>
                      )}

                      <div className="iq-div">
                        <Button className="iq-btn">{translateAmenties('Inquire Now')}</Button>
                      </div>
                    </>
                  )}
                </Form>
              </fieldset>
            </div>
          </div>
          <div className="col-md-5 right">
            <div className="topss">
              <img src={intro2} alt="profile" />
            </div>
            <div className="bottms">
              <div>
                <p>{i18next.t('We would not be where we are today')}.</p>
                <p className="nam"> Kai Stiberg, Eagle</p>
              </div>
            </div>
          </div>
        </div>
      </div>
    </>
  );
};
