import React, { useState } from 'react';
import * as z from 'zod';

import { Button, Dialog, Spinner } from '@/components/Elements';
import { Form } from '@/components/Form';

import { useNotificationStore } from '@/stores/notifications';
import { useAuth } from '@/lib/auth';
import { verifyOtp } from '@/api/profile/verifyOtp';
import { resendVerifyLink } from '@/api/profile/sendVerifyLink';
import { numbers } from '@/constants/numbers';
import i18next from 'i18next';

const schema = z.object({
  digit1: z.string(),
  digit2: z.string(),
  digit3: z.string(),
  digit4: z.string(),
});

type OtpValues = {
  digit1: string;
  digit2: string;
  digit3: string;
  digit4: string;
};

const OTPVerify = ({ isOpen, close }: { isOpen: boolean; close: () => void }) => {
  const { user } = useAuth();
  const [loading, setLoading] = useState(false);
  const [isResending, setResending] = useState(false);

  const doNotClose = () => {};

  const sendOtpAgain = async () => {
    setResending(true);
    await resendVerifyLink();
    useNotificationStore.getState().addNotification({
      title: 'Success',
      type: 'info',
      message: 'Otp has been sent to your email address',
    });
    setResending(false);
  };

  const resendClicked = () => {
    if (!isResending) {
      sendOtpAgain();
    }
  };

  const focus = (input: number) => (e: any) => {
    if (e.code === 'Backspace') {
      if (e.target.value.length === 0) {
        if (input > 2) {
          const item = document.getElementById(`input${input - 2}`);
          item?.focus();
        }
        if (input == 2) {
          const item = document.getElementById(`input${input - 1}`);
          item?.focus();
        }
      }
    } else {
      if (numbers.includes(e.code)) {
        if (input < 5) {
          const item = document.getElementById(`input${input}`);
          item?.focus();
        }
      }
    }
  };

  return (
    <div>
      <Dialog isOpen={isOpen} onClose={doNotClose}>
        <div className="inline-block align-top bg-white px-4 pb-4 text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full sm:p-6">
          <div className="row m-5">
            <div className="col-12 mt-3">
              <h2>We have sent a temporary code to your email</h2>
              <p className="mt-4 mb-0">Enter the code we sent to</p>
              <p>
                <b>{user?.email} </b>
              </p>
              <div className="my-4">
                <Form<OtpValues, typeof schema>
                  onSubmit={async (values) => {
                    try {
                      setLoading(true);
                      const payload = {
                        otp: `${values.digit1}${values.digit2}${values.digit3}${values.digit4}`,
                      };
                      await verifyOtp(payload);
                      useNotificationStore.getState().addNotification({
                        title: 'Success',
                        type: 'success',
                        message: 'Email verified successfully',
                      });
                      close();
                    } finally {
                      setLoading(false);
                    }
                  }}
                  schema={schema}
                >
                  {({ register }) => (
                    <div>
                      <div className="row digits">
                        <div className="col-2">
                          <input
                            id="input1"
                            onKeyUp={focus(2)}
                            type="number"
                            {...register('digit1')}
                            className="form-control"
                          />
                        </div>
                        <div className="col-2">
                          <input
                            onKeyUp={focus(3)}
                            id="input2"
                            type="number"
                            {...register('digit2')}
                            className="form-control"
                          />
                        </div>
                        <div className="col-2">
                          <input
                            id="input3"
                            onKeyUp={focus(4)}
                            type="number"
                            {...register('digit3')}
                            className="form-control"
                          />
                        </div>
                        <div className="col-2">
                          <input
                            id="input4"
                            onKeyUp={focus(5)}
                            type="number"
                            {...register('digit4')}
                            className="form-control"
                          />
                        </div>
                      </div>
                      <div className="d-flex my-4">
                        <p className="">
                          Didn&apos;t get the code?
                          <span
                            tabIndex={-1}
                            onKeyDown={resendClicked}
                            role="button"
                            onClick={resendClicked}
                            className="text-blue-500 cursor-pointer"
                          >
                            Send the code again
                          </span>
                        </p>
                        <div className="mt-1 text-blue-500">
                          {isResending && <Spinner size="sm" />}
                        </div>
                      </div>
                      <div className="my-4 d-flex">
                        <Button isLoading={loading} type="submit" className="w-50 me-3">
                          {i18next.t('Continue')}
                        </Button>
                        <Button onClick={close} variant="outline" className="w-45">
                          {i18next.t(`Cancel`)}
                        </Button>
                      </div>
                    </div>
                  )}
                </Form>
              </div>
            </div>
          </div>
        </div>
      </Dialog>
    </div>
  );
};

export default OTPVerify;
