import React, { useState } from 'react';
import * as z from 'zod';

import verified from '@/assets/verified.svg';
import { Button, Dialog } from '@/components/Elements';
import { Form, InputField } from '@/components/Form';

import { sendVerifyLink } from '@/api/profile/sendVerifyLink';
import { useNotificationStore } from '@/stores/notifications';
import { InputPhone } from '@/components/Form/InputPhone';
import { isValidMobile, isValidMobileMsg } from '@/utils/validation';

const schema = z.object({
  phone_number: z.custom(isValidMobile, isValidMobileMsg),
});

type VerifyValues = {
  phone_number: string;
};

const PhoneVerify = ({
  isOpen,
  close,
  onSuccess,
}: {
  isOpen: boolean;
  close: () => void;
  onSuccess: () => void;
}) => {
  const [loading, setLoading] = useState(false);

  return (
    <div>
      <Dialog size="large" isOpen={isOpen} onClose={close}>
        <div className="inline-block align-top bg-white px-4 pb-4 text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full sm:p-6">
          <div className="row my-5">
            <div className="col-5">
              <img src={verified} className="w-100 px-5 mt-3" alt="verified" />
            </div>
            <div className="col-6 mt-3">
              <h2>Safety First</h2>
              <p className="my-4">
                To keep everyone safe and happy on REEIPO, only verified users can connect with
                sellers. We can verify your profile by adding your phone number. Use the box below
                to start.
              </p>
              <div className="my-4">
                <Form<VerifyValues, typeof schema>
                  onSubmit={async (values) => {
                    try {
                      setLoading(true);
                      await sendVerifyLink(values);
                      useNotificationStore.getState().addNotification({
                        title: 'Success',
                        type: 'info',
                        message: 'Otp has been sent to your email address',
                      });
                      onSuccess();
                    } finally {
                      setLoading(false);
                    }
                  }}
                  schema={schema}
                >
                  {({ register, formState }) => (
                    <>
                      <InputPhone
                        floating
                        error={formState.errors['phone_number']}
                        registration={register('phone_number')}
                      />

                      <div className="my-4 d-flex">
                        <Button variant="outline" onClick={close} className="w-45 me-2">
                          Maybe Later
                        </Button>
                        <Button isLoading={loading} type="submit" className="w-55">
                          Send Verification Code
                        </Button>
                      </div>
                    </>
                  )}
                </Form>
              </div>
            </div>
          </div>
        </div>
      </Dialog>
    </div>
  );
};

export default PhoneVerify;
