import React, { useState } from 'react';
import * as z from 'zod';

// import plus from '@/assets/plus.png';

import { ContentLayout } from '@/components/Layout';
import { useSelectFile } from '@/hooks/useSelectFile';
import { Form, InputField, SelectField, TextAreaField } from '@/components/Form';
import ProfileLoader from '@/components/Elements/Loader/ProfileLoader';
import { Button } from '@/components/Elements';
import { uploadImage } from '@/api/uploadImage';
import { editBussinessSale } from '@/api/bussinessForSale/editBussinessForSale';
import { useNotificationStore } from '@/stores/notifications';
import { useNavigate, useParams } from 'react-router-dom';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
// import { useBsfDetails } from '@/api/bussinessForSale/getBFS';
import { useCommonDetails } from '@/api/commonDetail/getCommonDetail';
import openingHours from '@/constants/openingHours';
import companyCategory from '@/constants/companyCategory';
import { Business } from '@/types';
import i18next from 'i18next';

const schema = z.object({
  company_name: z.string().min(1, 'Please enter company name'),
  company_details: z.string().min(1, 'Please enter company details'),
  purchase_price: z.string().min(1, 'Please enter purchase price'),
  total_turnover: z.string().min(1, 'Please enter total turnover'),
  operating_profit: z.string().min(1, 'Please enter operating profit'),
  salary_percentage: z.string().min(1, 'Please enter Sallary Percentage'),
  company_established: z.string().min(1, 'Please enter company established'),
  number_of_employes: z.string().min(1, 'Please enter number of employees'),
  company_email: z
    .string()
    .min(1, 'Please enter company email address')
    .email('Please enter a valid email address'),
  company_telephone: z
    .string()
    .min(1, 'Please enter company phone')
    .regex(/^[-0-9+]+$/, 'Phone should contain numbers only!'),
  company_category: z.string().min(1, 'Please enter company category'),
  opening_hours: z.string().min(1, 'Please enter opening hours'),
  company_facebook_link: z.string().optional(),
  company_twitter_link: z.string().optional(),
  company_youtube_link: z.string().optional(),
  company_linkedin_link: z.string().optional(),
  company_vision: z.string().optional(),
  note_to_purchase: z.string().optional(),
});

type editBussinessSale = {
  price: any;
  customer_name: any;
  property_id: string;
  image_data: string;
  videos: string;
  company_name: string;
  company_established: string;
  number_of_employes: string;
  company_email: string;
  company_telephone: string;
  company_category: string;
  opening_hours: string;
  company_twitter_link: string;
  company_facebook_link: string;
  company_youtube_link: string;
  company_linkedin_link: string;
  company_details: string;
  company_vision: string;
  note_to_purchase: string;
  purchase_price: string;
  total_turnover: string;
  operating_profit: string;
  salary_percentage: string;
};

export const EditBussinessSale = () => {
  const { id } = useParams();

  const { data, isLoading, isRefetching, refetch } = useCommonDetails({ id: id ?? '1' });
  const user = data?.data ?? ({} as Business);
  const [loading, setLoading] = useState(false);
  const picOne = user?.image_data;
  const videoOne = user?.videos;
  const navigate = useNavigate();
  const { file, preview, handleChange } = useSelectFile();
  const {
    file: videoFile,
    preview: videoPreview,
    handleChange: handleVideoChange,
    cover,
  } = useSelectFile();
  // mmm

  const handleSubmit = async (values: editBussinessSale) => {
    try {
      setLoading(true);
      let imageUrl = picOne,
        videoUrl = videoOne;

      if (file) {
        const response = await uploadImage({ file });
        imageUrl = response.url;
      }

      if (videoFile) {
        const response = await uploadImage({ file: videoFile });
        videoUrl = response.url;
      }
      await editBussinessSale({
        ...values,
        videos: videoUrl,
        image_data: imageUrl,
        property_id: `${id}`,
      });
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'Submitted successfully!',
      });
      refetch();
      navigate('/user/my-listing');
    } finally {
      setLoading(false);
    }
  };

  return (
    <ContentLayout title="Create Profile">
      <button onClick={() => navigate(-1)}>
        <i className="fa-regular fa-circle-left go-back-button"></i>
      </button>
      <BackdropLoader open={loading} />
      <button onClick={() => navigate(-1)}>
        <i className="fa-regular fa-circle-left go-back-button"></i>
      </button>

      <div className="container mt-4">
        <h3>Edit : </h3>
        <h5 className="ms-2 my-4">Business For Sale Detail</h5>
        {isLoading || isRefetching ? (
          <ProfileLoader />
        ) : (
          <Form<editBussinessSale, typeof schema>
            onSubmit={handleSubmit}
            schema={schema}
            options={{
              defaultValues: user,
            }}
          >
            {({ register, formState }) => (
              <div className="profile-box mb-5">
                <div className="d-flex">
                  <div>
                    <h5>Photo</h5>
                    <label htmlFor="fileInput">
                      <div className="image-box">
                        {preview ? (
                          <img src={preview} alt="" className="profileImage" />
                        ) : (
                          <img src={picOne} alt="" width="60" />
                        )}
                      </div>
                    </label>
                    <input
                      accept="image/*"
                      onChange={handleChange}
                      type="file"
                      id="fileInput"
                      className="d-none"
                    />
                  </div>
                  <div className="ms-4">
                    <h5>Video</h5>
                    <label htmlFor="videoInput">
                      <div className="image-box">
                        {videoPreview ? (
                          <img src={cover?.toString() ?? ''} alt="" className="profileImage" />
                        ) : (
                          <img src={videoOne} alt="" width="60" />
                        )}
                      </div>
                    </label>
                    <input
                      accept="video/mp4,video/x-m4v,video/*"
                      onChange={handleVideoChange}
                      type="file"
                      id="videoInput"
                      className="d-none"
                    />
                  </div>
                </div>
                <div className="row mt-4">
                  <div className="col-12 col-md-6 mt-4">
                    <InputField
                      type="text"
                      label={i18next.t('Company name')}
                      error={formState.errors['company_name']}
                      registration={register('company_name')}
                      placeholder={i18next.t('Company name')}
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-4">
                    <InputField
                      type="number"
                      label="Company established (In Years)"
                      error={formState.errors['company_established']}
                      registration={register('company_established')}
                      placeholder="Company established (In Years)"
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-4">
                    <InputField
                      type="number"
                      min={0}
                      label="Number of employees"
                      error={formState.errors['number_of_employes']}
                      registration={register('number_of_employes')}
                      placeholder="Number of employees"
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-4">
                    <InputField
                      type="email"
                      label="Company Email Address"
                      error={formState.errors['company_email']}
                      registration={register('company_email')}
                      placeholder="Company Email Address"
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-4">
                    <InputField
                      type="text"
                      label="Company Telephone"
                      error={formState.errors['company_telephone']}
                      registration={register('company_telephone')}
                      placeholder="Company Telephone"
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-4">
                    <SelectField
                      label="Company Category"
                      error={formState.errors['company_category']}
                      registration={register('company_category')}
                      placeholder="Company Category"
                      options={Object.entries(companyCategory).map(([key, value]) => ({
                        label: value,
                        value: key,
                      }))}
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-4">
                    <SelectField
                      label="Opening Hours"
                      error={formState.errors['opening_hours']}
                      registration={register('opening_hours')}
                      placeholder="Opening Hours"
                      options={Object.entries(openingHours).map(([key, value]) => ({
                        label: value,
                        value: key,
                      }))}
                    />
                  </div>
                  <p className="mt-4 btn-heavy">Company Social Media</p>
                  <div className="col-12 col-md-6">
                    <InputField
                      type="text"
                      label="Facebook"
                      error={formState.errors['company_facebook_link']}
                      registration={register('company_facebook_link')}
                      placeholder="Facebook"
                    />
                  </div>
                  <div className="col-6">
                    <InputField
                      type="text"
                      label="Twitter"
                      error={formState.errors['company_twitter_link']}
                      registration={register('company_twitter_link')}
                      placeholder="Twitter"
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-2">
                    <InputField
                      type="text"
                      label={i18next.t('Youtube')}
                      error={formState.errors['company_youtube_link']}
                      registration={register('company_youtube_link')}
                      placeholder={i18next.t('Youtube')}
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-2">
                    <InputField
                      type="text"
                      label="Linkedin"
                      error={formState.errors['company_linkedin_link']}
                      registration={register('company_linkedin_link')}
                      placeholder="Linkedin"
                    />
                  </div>
                  <div className="col-12 mt-4">
                    <TextAreaField
                      rows={4}
                      error={formState.errors['company_details']}
                      registration={register('company_details')}
                      label={i18next.t('Company Details')}
                    />
                  </div>
                  <div className="col-12 mt-4">
                    <TextAreaField
                      rows={4}
                      error={formState.errors['company_vision']}
                      registration={register('company_vision')}
                      label="Company Vision"
                    />
                  </div>
                  <div className="col-12 mt-4">
                    <TextAreaField
                      rows={4}
                      error={formState.errors['note_to_purchase']}
                      registration={register('note_to_purchase')}
                      label="Important note to purchases"
                    />
                  </div>
                  <p className="mt-4 btn-heavy">Company Information</p>
                  <div className="col-12 col-md-6 mt-2">
                    <InputField
                      type="number"
                      label={i18next.t('Purchase price')}
                      error={formState.errors['purchase_price']}
                      registration={register('purchase_price')}
                      placeholder={i18next.t('Purchase price')}
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-2">
                    <InputField
                      type="number"
                      label="Total Turnover"
                      error={formState.errors['total_turnover']}
                      registration={register('total_turnover')}
                      placeholder="Total Turnover"
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-2">
                    <InputField
                      type="number"
                      label="Operating Profit"
                      error={formState.errors['operating_profit']}
                      registration={register('operating_profit')}
                      placeholder="Operating Profit"
                    />
                  </div>
                  <div className="col-12 col-md-6 mt-2">
                    <InputField
                      type="number"
                      label="Salary percentage"
                      error={formState.errors['salary_percentage']}
                      registration={register('salary_percentage')}
                      placeholder="Salary percentage"
                    />
                  </div>
                  <div className="d-flex justify-content-end mt-4">
                    <Button type="submit" block isLoading={loading}>
                      Submit
                    </Button>
                  </div>
                </div>
              </div>
            )}
          </Form>
        )}
      </div>
    </ContentLayout>
  );
};
