import React, { useState } from 'react';
import * as z from 'zod';

import plus from '@/assets/plus.png';

import { ContentLayout } from '@/components/Layout';
import { useSelectFile } from '@/hooks/useSelectFile';
import { Form, InputField, SelectField, TextAreaField } from '@/components/Form';
import { Button } from '@/components/Elements';
import { useNotificationStore } from '@/stores/notifications';
import { useNavigate, useParams } from 'react-router-dom';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import propertyType from '@/constants/propertyType';
import { Housing } from '@/types';
import { uploadImage } from '@/api/uploadImage';
import { useHousingIdDetails } from '@/api/housing/housingGetId';
import { editHousing } from '@/api/housing/editHousing';
import EditPageLoader from '@/components/Elements/Loader/EditPageLoader';
import { InputPhone } from '@/components/Form/InputPhone';
import { isValidMobile, isValidMobileMsg } from '@/utils/validation';
import i18next from 'i18next';
const schema = z.object({
  first_name: z.string().min(1, 'Please enter First Name'),
  last_name: z.string().min(1, 'Please enter Last Name'),
  gender: z.string().min(1, 'Please enter Gender'),
  dob: z.string().min(1, 'Please enter DOB'),
  address: z.string().min(1, 'Please enter Address'),
  country: z.string().min(1, 'Please enter Nationality'),
  no_of_resident: z.string().min(1, 'Please enter Number of Resident.'),
  phone_number: z.custom(isValidMobile, isValidMobileMsg),

  email: z
    .string()
    .min(1, 'Please enter company email address')
    .email('Please enter a valid email address'),
  about: z.string().optional(),
  max_month_price: z.string().min(1, 'Please enter Max month rent .'),
  property_type: z.string().min(1, 'Please Select Property type'),
  wanted_from: z.string().min(1, 'Please enter house wanted for rent date'),
  pet: z.string().min(1, 'Please enter if pet any.'),
  location: z.string().min(1, 'Please enter your Location'),
});

type editHousing = {
  first_name: string;
  last_name: string;
  gender: string;
  dob: string;
  address: string;
  country: string;
  phone_number: string;
  email: string;
  about: string;
  max_month_price: string;
  opening_hours: string;
  property_type: string;
  wanted_from: string;
  s;
  number_of_resident: string;
  no_of_resident: string;
  location: string;
  pet: string;
  rent_id: string;
};

export const EditHouseDetail = () => {
  const { id } = useParams();
  const { data, isLoading, isRefetching, refetch } = useHousingIdDetails({ id: id ?? '1' });
  const user = data?.data ?? ({} as Housing);
  const [loading, setLoading] = useState(false);

  const navigate = useNavigate();
  const { file, preview, handleChange } = useSelectFile();
  const {
    file: videoFile,
    preview: videoPreview,
    handleChange: handleVideoChange,
    cover,
  } = useSelectFile();

  const handleSubmit = async (values: editHousing) => {
    try {
      setLoading(true);
      let imageUrl = '',
        videoUrl = '';

      if (file) {
        const response = await uploadImage({ file });
        imageUrl = response.url;
      }

      if (videoFile) {
        const response = await uploadImage({ file: videoFile });
        videoUrl = response.url;
      }
      await editHousing({
        ...values,
        videos: videoUrl,
        image: imageUrl,
        rent_id: `${id}`,
        property_id: `${id}`,
      });
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'Submitted successfully!',
      });
      refetch();
      navigate('/user/my-listing');
    } finally {
      setLoading(false);
    }
  };

  return (
    <ContentLayout title="Edit House for Rent">
      <BackdropLoader open={loading} />
      <button onClick={() => navigate(-1)}>
        <i className="fa-regular fa-circle-left go-back-button"></i>
      </button>

      <div className="container mt-4">
        <h3>Edit : </h3>
        <h5 className="ms-2 my-4">Edit House for Rent details</h5>
        {isLoading || isRefetching ? (
          <EditPageLoader />
        ) : (
          <Form<editHousing, typeof schema>
            onSubmit={handleSubmit}
            schema={schema}
            options={{
              defaultValues: user,
            }}
          >
            {({ register, formState }) => (
              <div className="profile-box mb-5">
                <div className="d-flex">
                  <div>
                    <h5>Photo</h5>
                    <label htmlFor="fileInput">
                      <div className="image-box">
                        {preview ? (
                          <img src={preview} alt="" className="profileImage" />
                        ) : (
                          <img src={plus} alt="" width="60" />
                        )}
                      </div>
                    </label>
                    <input
                      accept="image/*"
                      onChange={handleChange}
                      type="file"
                      id="fileInput"
                      className="d-none"
                    />
                  </div>
                  <div className="ms-4">
                    <h5>Video</h5>
                    <label htmlFor="videoInput">
                      <div className="image-box">
                        {videoPreview ? (
                          <img src={cover?.toString() ?? ''} alt="" className="profileImage" />
                        ) : (
                          <img src={plus} alt="" width="60" />
                        )}
                      </div>
                    </label>
                    <input
                      accept="video/mp4,video/x-m4v,video/*"
                      onChange={handleVideoChange}
                      type="file"
                      id="videoInput"
                      className="d-none"
                    />
                  </div>
                </div>
                <div className="row mt-4">
                  <div className="col-6 mt-4">
                    <InputField
                      type="text"
                      label="First Name"
                      error={formState.errors['first_name']}
                      registration={register('first_name')}
                      placeholder="First Name"
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <InputField
                      type="text"
                      label="Last Name"
                      error={formState.errors['last_name']}
                      registration={register('last_name')}
                      placeholder="Last Name"
                    />
                  </div>

                  <div className="col-6 mt-4">
                    <InputField
                      type="text"
                      label="Gender"
                      error={formState.errors['gender']}
                      registration={register('gender')}
                      placeholder="Gender"
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <InputField
                      type="date"
                      label="D.O.B."
                      error={formState.errors['dob']}
                      registration={register('dob')}
                      placeholder="D.O.B."
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <InputField
                      type="text"
                      label="address"
                      error={formState.errors['address']}
                      registration={register('address')}
                      placeholder="Address"
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <InputField
                      type="text"
                      label="Nationality"
                      error={formState.errors['country']}
                      registration={register('country')}
                      placeholder="Nationality"
                    />
                  </div>

                  <div className="col-6 mt-4">
                    <SelectField
                      label={i18next.t('Property Type')}
                      error={formState.errors['property_type']}
                      registration={register('property_type')}
                      options={Object.entries(propertyType).map(([key, value]) => ({
                        label: value,
                        value: key,
                      }))}
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <InputPhone
                      floating
                      error={formState.errors['phone_number']}
                      registration={register('phone_number')}
                    />
                  </div>
                  <div className="col-6 mt-2">
                    <InputField
                      type="email"
                      label={i18next.t('Email')}
                      error={formState.errors['email']}
                      registration={register('email')}
                    />
                  </div>
                  <div className="col-6 mt-2">
                    <InputField
                      type="number"
                      label="Max Month Price"
                      error={formState.errors['max_month_price']}
                      registration={register('max_month_price')}
                      placeholder="Max Month Price"
                    />
                  </div>

                  <div className="col-6 mt-2">
                    <InputField
                      type="date"
                      label={i18next.t('Wanted From')}
                      error={formState.errors['wanted_from']}
                      registration={register('wanted_from')}
                      placeholder={i18next.t('Wanted From')}
                    />
                  </div>
                  <div className="col-6 mt-2">
                    <InputField
                      type="number"
                      label="No of Resident"
                      error={formState.errors['no_of_resident']}
                      registration={register('no_of_resident')}
                      placeholder="No Of Resident"
                    />
                  </div>
                  <div className="col-6 mt-2">
                    <InputField
                      type="text"
                      label={i18next.t('Pet')}
                      error={formState.errors['pet']}
                      registration={register('pet')}
                      placeholder={i18next.t('Pet')}
                    />
                  </div>
                  <div className="col-6 mt-2">
                    <InputField
                      type="text"
                      label={i18next.t('Location')}
                      error={formState.errors['location']}
                      registration={register('location')}
                    />
                  </div>
                  <div className="col-12 mt-4">
                    <TextAreaField
                      rows={4}
                      error={formState.errors['about']}
                      registration={register('about')}
                      label="About"
                    />
                  </div>
                  <div className="d-flex justify-content-end mt-4">
                    <Button type="submit" block isLoading={loading}>
                      Submit
                    </Button>
                  </div>
                </div>
              </div>
            )}
          </Form>
        )}
      </div>
    </ContentLayout>
  );
};
