import React, { useEffect, useState } from 'react';
import * as z from 'zod';
import { ContentLayout } from '@/components/Layout';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import UserHeadBar from '../components/UserHeadBar';
import '../user.css';
import { Button, Drawer } from '@/components/Elements';
import ServiceItem from '../components/ServiceItem';
import { Form, InputField } from '@/components/Form';
import { useListing } from '@/api/myListing/myListing';
import MyListingLoader from '@/components/Elements/Loader/MyListingLoader';
import { useNavigate } from 'react-router-dom';
import { useDisclosure } from '@/hooks/useDisclosure';
import {
  FormControl,
  FormControlLabel,
  FormLabel,
  InputLabel,
  MenuItem,
  Radio,
  RadioGroup,
} from '@mui/material';
import Select, { SelectChangeEvent } from '@mui/material/Select';
import { useFilterListing } from '@/api/myListing/getFiltersListing';
import { SaleVerifyContract } from '@/features/misc/routes/SaleDetail/SalePayment/SaleVerifyContract';
import i18next from 'i18next';

type SearchValues = {
  search: string;
};

const schema = z.object({
  search: z.string().min(1, 'Please enter first name'),
});

export const MyListing = () => {
  const navigate = useNavigate();
  const { close, open, isOpen } = useDisclosure();
  const [loading] = useState(false);
  const [timeSort, setTimeSort] = useState('');
  const [priceSort, setPriceSort] = useState('');
  const [clear, setClear] = useState<boolean>(false);
  const [filter, setFilter] = useState<boolean>(true);
  const [sale, setSale] = useState<string>('');
  const [type, setType] = useState<string>('');

  const [dRefetch, setDeleteRefetch] = useState<boolean>();

  const {
    data: listing,
    isLoading,
    refetch,
  } = useFilterListing({
    page: '1',
    filter_data: filter,
    property_type: type ? type : '',
    sort_price: priceSort ? priceSort : '',
    sort_date: timeSort ? timeSort : '',
  });

  useEffect(() => {
    refetch();
    setDeleteRefetch(false);
  }, [dRefetch, refetch]);

  const clearFilter = () => {
    setClear(true);
    setFilter(false);
    setType('');
    setTimeSort('');
    setPriceSort('');
    setSale('');
    setTimeout(() => {
      refetch();
    }, 2000);
    close();
  };

  const handleSelectChange = (event: SelectChangeEvent) => {
    const selectedSale = event.target.value as string;
    setSale(selectedSale);
    setType(selectedSale);
  };

  const handleSaveSearch = () => {
    setFilter(true);
    setTimeout(() => {
      refetch();
    }, 2000);
    close();
  };

  useEffect(() => {
    refetch();
  }, [refetch, filter]);

  return (
    <ContentLayout title="My Listing">
      <BackdropLoader open={loading} />
      <UserHeadBar content="These are your listings. The more information you add, the better REEIPO can serve your needs." />
      <button onClick={() => navigate(-1)}>
        <i className="fa-regular fa-circle-left go-back-button"></i>
      </button>

      <div className="container listingContainer">
        <div className="d-flex justify-content-between listing-prf">
          <h3>My Listing</h3>
          <div className="d-flex align-items-center">
            {/* <Form<SearchValues, typeof schema> onSubmit={() => null} schema={schema}>
              {({ register, formState }) => (
                <InputField
                  label="Search"
                  error={formState.errors['search']}
                  registration={register('search')}
                  placeholder="First name"
                />
              )}
            </Form> */}
            <Button
              className="ms-3 filterBtn"
              variant="outlineDark"
              startIcon={<i className="fa-solid fa-sliders" />}
              onClick={open}
            >
              {i18next.t('Filters')}
            </Button>
          </div>
          <Drawer
            isOpen={isOpen}
            onClose={close}
            title={i18next.t('Filters')}
            size="sm"
            renderFooter={() => (
              <>
                <Button
                  onClick={() => clearFilter()}
                  variant="primary"
                  className="w-100 justify-center"
                  size="md"
                >
                  {i18next.t('Clear')}
                </Button>
                <Button
                  onClick={() => handleSaveSearch()}
                  variant="primary"
                  className="w-100 justify-center"
                  size="md"
                >
                  {i18next.t('Apply')}
                </Button>
              </>
            )}
          >
            <div>
              <FormControl>
                <FormLabel id="demo-radio-buttons-group-label">
                  {i18next.t('Sort data according to price')}
                </FormLabel>
                <RadioGroup
                  aria-labelledby="demo-radio-buttons-group-label"
                  name="radio-buttons-group"
                >
                  <FormControlLabel
                    value="ascending"
                    control={<Radio />}
                    onClick={() => setPriceSort('ascending')}
                    label={i18next.t('Low to high')}
                    checked={priceSort === 'ascending'}
                  />
                  <FormControlLabel
                    value="desending"
                    control={<Radio />}
                    onClick={() => setPriceSort('desending')}
                    label={i18next.t('High to low')}
                    checked={priceSort === 'desending'}
                  />
                </RadioGroup>
              </FormControl>

              {/* -----------------------__? */}
              <hr />
              <p>Select Property Type</p>
              <div className="row">
                <div>
                  <FormControl fullWidth>
                    <InputLabel id="demo-simple-select-label">Property Type</InputLabel>
                    <Select
                      labelId="demo-simple-select-label"
                      id="demo-simple-select"
                      label={i18next.t('Property Type')}
                      value={sale}
                      onChange={handleSelectChange}
                    >
                      <MenuItem value={''}>{i18next.t('All Properties')}</MenuItem>
                      <MenuItem value={'SaleProperty'}>{i18next.t('Property For Sale')}</MenuItem>
                      <MenuItem value={'RentProperty'}>{i18next.t('Property For Rent')}</MenuItem>
                      <MenuItem value={'BusinessSale'}>{i18next.t('Business For Sale')}</MenuItem>
                      <MenuItem value={'HouseRent'}>{i18next.t('House for rent')}</MenuItem>
                      <MenuItem value={'InvestmentProject'}>
                        {i18next.t('Investment Project')}
                      </MenuItem>
                      <MenuItem value={'InvestmentCompany'}>
                        {i18next.t('Investment Company')}
                      </MenuItem>
                      <MenuItem value={'PropertyProject'}>{i18next.t('Property Project')}</MenuItem>
                    </Select>
                  </FormControl>
                </div>
              </div>

              <hr />

              <FormControl>
                <FormLabel id="demo-radio-buttons-group-label">Date Created</FormLabel>
                <RadioGroup
                  aria-labelledby="demo-radio-buttons-group-label"
                  name="radio-buttons-group"
                >
                  <FormControlLabel
                    value="oldest"
                    control={<Radio />}
                    onClick={() => setTimeSort('oldest')}
                    label={i18next.t('Oldest')}
                    checked={timeSort === 'oldest'}
                  />
                  <FormControlLabel
                    value="latest"
                    control={<Radio />}
                    onClick={() => setTimeSort('latest')}
                    label={i18next.t('Latest')}
                    checked={timeSort === 'latest'}
                  />
                </RadioGroup>
              </FormControl>
            </div>
          </Drawer>
        </div>
        <div className="row">
          {listing?.length == 0 ? (
            <h4 className="makeCenter">{i18next.t('No listing found !')}</h4>
          ) : (
            <>
              {isLoading ? (
                <MyListingLoader />
              ) : (
                <>
                  {listing?.map((item, index) => (
                    <ServiceItem
                      showEditIcon={true}
                      key={index}
                      item={item}
                      showDeleteIcon={true}
                      setDeleteRefetch={setDeleteRefetch}
                      showHeartFavIcon={false}
                      showHeartIcon={false}
                    />
                  ))}
                </>
              )}
            </>
          )}
        </div>
      </div>
    </ContentLayout>
  );
};
