import React, { useState } from 'react';
import * as z from 'zod';
import { Country } from 'country-state-city';

import plus from '@/assets/plus.png';
import { Form, InputField, RadioButtonField, SelectField, TextAreaField } from '@/components/Form';

import { ContentLayout } from '@/components/Layout';
import experiences from '@/constants/experiences';
import companyType from '@/constants/companyType';
import { Button } from '@/components/Elements';
import { uploadImage } from '@/api/uploadImage';
import { completeProfile } from '@/api/profile/completeProfile';
import { formatString } from '@/utils/format';
import { useNotificationStore } from '@/stores/notifications';
import { useAuth } from '@/lib/auth';
import { useNavigate } from 'react-router-dom';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useUserStore } from '@/stores/user';
import i18next from 'i18next';

const schema = z
  .object({
    first_name: z.string().min(1, 'Please enter first name'),
    last_name: z.string().min(1, 'Please enter last name'),
    email: z.string().min(1, 'Please enter email address'),
    gender: z.string().min(1, 'Please select gender'),
    address: z.string().min(1, 'Please enter address'),
    dob: z.string().min(1, 'Please select date of birth'),
    country: z.string().min(1, 'Please select country'),
    user_type: z.string().min(1, 'Please select user type'),
    your_self: z.string().min(1, 'Please enter description'),
    past_experience: z.string().min(1, 'Please select past experience'),
    user_level: z.string().min(1, 'Please select one option'),
    company_name: z.string().optional(),
    company_url: z.string().optional(),
    company_slogan: z.string().optional(),
    company_email: z.string().optional(),
    company_telephone: z.string().optional(),
    company_detail: z.string().optional(),
    company_category: z.string().optional(),
    facebook_link: z.string().optional(),
    twitter_link: z.string().optional(),
    youtube_link: z.string().optional(),
    linkedin_link: z.string().optional(),
  })
  .superRefine((entry: any, ctx) => {
    if (entry.user_level === 'company') {
      [
        'company_name',
        'company_url',
        'company_slogan',
        'company_email',
        'company_telephone',
        'company_detail',
        'company_category',
      ].forEach((i) => {
        if (!entry[i]) {
          ctx.addIssue({
            path: [i],
            code: 'custom',
            message: `Please enter ${formatString(i)}`,
          });
        }
      });
    }
  });

type CompleteProfileValues = {
  first_name: string;
  last_name: string;
  email: string;
  gender: string;
  address: string;
  dob: string;
  country: string;
  user_type: string;
  your_self: string;
  past_experience: string;
  user_level: string;
  company_name: string;
  company_url: string;
  company_slogan: string;
  company_email: string;
  company_telephone: string;
  company_detail: string;
  twitter_link: string;
  facebook_link: string;
  youtube_link: string;
  linkedin_link: string;
  company_category: string;
};

const allCountries = Country.getAllCountries().map((i) => ({
  label: i.name,
  value: i.name,
}));

export const CompleteProfile = () => {
  const [file, setFile] = useState<File | null>(null);
  const [profilePicture, setPicturePreview] = useState<string>('');
  const [userLevel, setUserLevel] = useState('private_user');
  const [loading, setLoading] = useState(false);
  const { user, refetchUser } = useAuth();
  const navigate = useNavigate();

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      setPicturePreview(URL.createObjectURL(e.target.files[0]));
      setFile(e.target.files[0]);
    }
  };

  const handleLevelChange = (e: any) => {
    setUserLevel(e.target.value);
  };

  const handleSubmit = async (values: CompleteProfileValues) => {
    try {
      setLoading(true);
      let url = '';
      if (file) {
        const response = await uploadImage({ file });
        url = response.url;
      }
      await completeProfile({ ...values, profile_image: url });
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'Profile completed successfully!',
      });
      await refetchUser();
      if (!user?.phoneVerified) {
        useUserStore.getState().show();
      }
      navigate('/');
    } finally {
      setLoading(false);
    }
  };

  return (
    <ContentLayout title="Create Profile">
      <BackdropLoader open={loading} />

      <button onClick={() => navigate(-1)}>
        <i className="fa-regular fa-circle-left go-back-button"></i>
      </button>
      <div className="container mt-4">
        <h3>Create Profile</h3>
        <div className="profile-box">
          <h5>Profile Photo</h5>
          <label htmlFor="fileInput">
            <div className="image-box">
              {profilePicture ? (
                <img src={profilePicture} alt="" className="profileImage" />
              ) : (
                <img src={plus} alt="" width="60" />
              )}
            </div>
          </label>
          <input onChange={handleChange} type="file" id="fileInput" className="d-none" />
          <Form<CompleteProfileValues, typeof schema> onSubmit={handleSubmit} schema={schema}>
            {({ register, formState }) => (
              <>
                <div className="row mt-4">
                  <div className="col-6 mt-4">
                    <InputField
                      type="text"
                      label="First Name"
                      error={formState.errors['first_name']}
                      registration={register('first_name', {
                        value: user?.first_name,
                      })}
                      placeholder="First Name"
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <InputField
                      type="text"
                      label="Last Name"
                      error={formState.errors['last_name']}
                      registration={register('last_name', {
                        value: user?.last_name,
                      })}
                      placeholder="First Name"
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <SelectField
                      label="Gender"
                      error={formState.errors['gender']}
                      registration={register('gender')}
                      placeholder="Gender"
                      options={['Male', 'Female'].map((o) => ({
                        label: o,
                        value: o.toLowerCase(),
                      }))}
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <InputField
                      type="text"
                      label="Address"
                      error={formState.errors['address']}
                      registration={register('address')}
                      placeholder="Address"
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <InputField
                      type="date"
                      label="Date of birth"
                      error={formState.errors['dob']}
                      registration={register('dob', {
                        value: user?.dob,
                      })}
                      placeholder="Date of birth"
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <SelectField
                      label={i18next.t('Country')}
                      options={allCountries}
                      error={formState.errors['country']}
                      registration={register('country')}
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <SelectField
                      label="User Type"
                      error={formState.errors['user_type']}
                      registration={register('user_type')}
                      placeholder="User Type"
                      options={['Landlord', 'Tenant', 'Property Developer', 'Other'].map((o) => ({
                        label: o,
                        value: o.toLowerCase(),
                      }))}
                    />
                  </div>
                  <div className="col-6 mt-4">
                    <InputField
                      type="email"
                      label={i18next.t('Email')}
                      error={formState.errors['email']}
                      registration={register('email', {
                        value: user?.email,
                      })}
                    />
                  </div>
                  <div className="col-12 mt-4">
                    <TextAreaField
                      rows={5}
                      label="Tell About Your Self"
                      error={formState.errors['your_self']}
                      registration={register('your_self')}
                    />
                  </div>

                  <div className="col-6 mt-4">
                    <SelectField
                      label="Experience"
                      error={formState.errors['past_experience']}
                      registration={register('past_experience')}
                      placeholder="User Type"
                      options={experiences.map((o) => ({
                        label: o,
                        value: o,
                      }))}
                    />
                  </div>
                  <div className="col-6 mt-4 d-flex justify-content-around checkboxes">
                    <RadioButtonField
                      value="company"
                      registration={register('user_level', {
                        onChange: handleLevelChange,
                        value: 'private_user',
                      })}
                      error={formState.errors['user_level']}
                      label="Company"
                    />
                    <RadioButtonField
                      value="private_user"
                      registration={register('user_level', {
                        onChange: handleLevelChange,
                        value: 'private_user',
                      })}
                      error={formState.errors['user_level']}
                      label="Private person"
                    />
                  </div>
                </div>
                {userLevel === 'company' && (
                  <div className="row mt-0">
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label={i18next.t('Company Name')}
                        error={formState.errors['company_name']}
                        registration={register('company_name')}
                        placeholder={i18next.t('Company Name')}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Company URL/Slug"
                        error={formState.errors['company_url']}
                        registration={register('company_url')}
                        placeholder="Company URL/Slug"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Slogan"
                        error={formState.errors['company_slogan']}
                        registration={register('company_slogan')}
                        placeholder="Slogan"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <SelectField
                        label="Company Category"
                        error={formState.errors['company_category']}
                        registration={register('company_category')}
                        placeholder="Company Category"
                        options={Object.entries(companyType).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Company Email Address"
                        error={formState.errors['company_email']}
                        registration={register('company_email')}
                        placeholder="Company Email Address"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Company Telephone"
                        error={formState.errors['company_telephone']}
                        registration={register('company_telephone')}
                        placeholder="Company Telephone"
                      />
                    </div>
                    <div className="col-12 mt-4">
                      <TextAreaField
                        label={i18next.t('Company Details')}
                        error={formState.errors['company_detail']}
                        registration={register('company_detail')}
                        rows={5}
                      />
                    </div>
                    <p className="mt-4                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          ">
                      Company Social Media
                    </p>
                    <div className="col-6">
                      <InputField
                        type="text"
                        label="Facebook"
                        error={formState.errors['facebook_link']}
                        registration={register('facebook_link')}
                        placeholder="Facebook"
                      />
                    </div>
                    <div className="col-6">
                      <InputField
                        type="text"
                        label="Twitter"
                        error={formState.errors['twitter_link']}
                        registration={register('twitter_link')}
                        placeholder="Twitter"
                      />
                    </div>
                    <div className="col-6 mt-2">
                      <InputField
                        type="text"
                        label={i18next.t('Youtube')}
                        error={formState.errors['youtube_link']}
                        registration={register('youtube_link')}
                        placeholder={i18next.t('Youtube')}
                      />
                    </div>
                    <div className="col-6 mt-2">
                      <InputField
                        type="text"
                        label="Linkedin"
                        error={formState.errors['linkedin_link']}
                        registration={register('linkedin_link')}
                        placeholder="Linkedin"
                      />
                    </div>
                  </div>
                )}
                <div className="row">
                  <div className="col-12">
                    <Button isLoading={loading} type="submit">
                      Submit
                    </Button>
                  </div>
                </div>
              </>
            )}
          </Form>
        </div>
      </div>
    </ContentLayout>
  );
};
