import React, { useEffect, useRef, useState } from 'react';
import * as z from 'zod';
import { useNavigate } from 'react-router-dom';
import { ContentLayout } from '@/components/Layout';
import { Form, InputField } from '@/components/Form';
import { useProfile, updateProfile } from '@/features/auth/api/getProfile';
import { User } from '@/types';
import ProfileLoader from '@/components/Elements/Loader/ProfileLoader';
import { Button, Image } from '@/components/Elements';
import { useSelectFile } from '@/hooks/useSelectFile';
import '../../user.css';
import { uploadImage } from '@/api/uploadImage';
import { useNotificationStore } from '@/stores/notifications';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useAuth } from '@/lib/auth';
import UserHeadBar from '../../components/UserHeadBar';
import { InputPhone } from '@/components/Form/InputPhone';
import { isValidMobile, isValidMobileMsg } from '@/utils/validation';
import GooglePlacesAutocomplete from 'react-google-places-autocomplete';
import axiosmod from 'axios';
import i18next from 'i18next';

const googleMapAPIkey = process.env.REACT_APP_GOOGLE_MAPS_API_KEY as string;

const schema = z
  .object({
    username: z.string().min(1, 'Please enter username'),
    first_name: z.string().min(1, 'Please enter first name'),
    last_name: z.string().min(1, 'Please enter last name'),
    email: z.string().optional(),
    phone_number: z.custom(isValidMobile, isValidMobileMsg),
    whatsapp_number: z.custom(isValidMobile, isValidMobileMsg),
    facebook_url: z.string().optional(),
    twitter_url: z.string().optional(),
    youtube_url: z.string().optional(),
    instagram_url: z.string().optional(),
    linkedin_url: z.string().optional(),
    pintrest_url: z.string().optional(),
  })
  .superRefine(({ username }, ctx) => {
    if (username.indexOf(' ') >= 0) {
      ctx.addIssue({
        path: ['username'],
        code: 'custom',
        message: 'User name should not contain spaces',
      });
    }
  });

type UpdateProfileValues = {
  username: string;
  first_name: string;
  last_name: string;
  email: string;
  address: string;
  whatsapp_number: string;
  phone_number: string;
  twitter: string;
  facebook_url: string;
  twitter_url: string;
  instagram_url: string;
  youtube_url: string;
  linkedin_url: string;
  pintrest_url: string;
};

export const UpdateProfile = () => {
  const { data, isLoading, isRefetching, refetch } = useProfile();
  const { refetchUser } = useAuth();
  const navigate = useNavigate();
  const inputFileRef = useRef<HTMLInputElement | null>(null);
  const { file, preview: picPreview, handleChange } = useSelectFile();
  const [loading, setLoading] = useState(false);

  const [address, setAddress] = useState<string>('');
  const [locationGoogle, setLocationGoogle] = useState();

  const [value, setValue] = useState(null);
  const [latitude, setlatitude] = useState('');
  const [longitude, setlongitude] = useState('');

  // ------------------ Location ---------------------

  const handleChangeLocation = async (value: any) => {
    const placeId = value.value.place_id;
    const { data } = await axiosmod.get(
      `https://maps.googleapis.com/maps/api/geocode/json?place_id=${placeId}&key=${googleMapAPIkey}`
    );
    const location = data?.results[0]?.geometry?.location;
    const addressData = data?.results[0]?.formatted_address;

    if (location) {
      setLocationGoogle(location);
      setValue(value);
    }
    if (addressData) {
      setAddress(addressData);
    }
  };

  useEffect(() => {
    if (locationGoogle) {
      setlatitude(locationGoogle?.lat);
      setlongitude(locationGoogle?.lng);
    }
  }, [locationGoogle]);

  const profile = data?.data ?? ({} as User);
  const handleSubmit = async (values: UpdateProfileValues) => {
    try {
      setLoading(true);
      let profile_image = profile?.profile_image ?? '';

      if (file) {
        const response = await uploadImage({ file });
        profile_image = response.url;
      }
      await updateProfile({
        ...values,
        address: address ? address : profile?.address,
        profile_image,
        latitude: latitude ? latitude : profile?.latitude,
        longitude: longitude ? longitude : profile?.longitude,
      });
      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'Profile updated successfully!',
      });
      refetchUser();
      refetch();
      navigate('/user/profile');
    } finally {
      setLoading(false);
    }
  };

  const changeClicked = () => {
    inputFileRef?.current?.click();
  };

  return (
    <ContentLayout title="Profile">
      <BackdropLoader open={loading} />

      <UserHeadBar content="Please fill out your account details. The more you add, the better we can customize REEIPO to suit your needs." />
      <button onClick={() => navigate(-1)}>
        <i className="fa-regular fa-circle-left go-back-button"></i>
      </button>
      <div className="container my-4">
        <h4>Account details</h4>
        <div className="profile-box ">
          {isLoading || isRefetching ? (
            <ProfileLoader />
          ) : (
            <Form<UpdateProfileValues, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: profile,
              }}
            >
              {({ register, formState }) => (
                <>
                  <div className="row align-items-center social-prf">
                    <div className="col-12 col-md-12 col-md-3">
                      <h5>Username</h5>
                    </div>
                    <div className="col-12 col-md-12 col-md-9">
                      <InputField
                        type="text"
                        label="Username"
                        error={formState.errors['username']}
                        registration={register('username')}
                        placeholder={i18next.t('User Name')}
                      />
                    </div>
                  </div>
                  <div className="row align-items-center social-prf">
                    <div className="col-12 col-md-12 col-md-3">
                      <h5>First Name</h5>
                    </div>
                    <div className="col-12 col-md-12 col-md-9">
                      <InputField
                        type="text"
                        label="First Name"
                        error={formState.errors['first_name']}
                        registration={register('first_name')}
                        placeholder="First Name"
                      />
                    </div>
                  </div>
                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Last Name</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputField
                        type="text"
                        label="Last Name"
                        error={formState.errors['last_name']}
                        registration={register('last_name')}
                        placeholder="Last Name"
                      />
                    </div>
                  </div>
                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>{i18next.t('Email Address')}</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputField
                        type="email"
                        label={i18next.t('Email Address')}
                        disabled
                        error={formState.errors['email']}
                        registration={register('email')}
                        placeholder={i18next.t('Email Address')}
                      />
                    </div>
                  </div>
                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Profile Picture</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      {picPreview ? (
                        <img alt="banner" src={picPreview} className="previewImage" />
                      ) : (
                        <Image src={profile?.profile_image} alt="" className="previewImage" />
                      )}
                      <Button onClick={changeClicked} variant="outline" size="sm" className="mt-2">
                        Change
                      </Button>
                      <input
                        ref={inputFileRef}
                        onChange={handleChange}
                        type="file"
                        id="fileInput"
                        className="d-none"
                      />
                    </div>
                  </div>
                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Telephone</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputPhone
                        floating
                        value={profile?.phone_number}
                        error={formState.errors['phone_number']}
                        registration={register('phone_number')}
                      />
                    </div>
                  </div>
                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Whatsapp Phone</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputPhone
                        floating
                        value={profile?.whatsapp_number}
                        error={formState.errors['whatsapp_number']}
                        registration={register('whatsapp_number')}
                      />
                    </div>
                  </div>
                  <div className="row align-items-center">
                    <div className="col-12 col-md-3">
                      <h5>Address</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      {/* <InputField
                        type="text"
                        label="Address"
                        error={formState.errors['address']}
                        registration={register('address')}
                        placeholder="Address"
                      /> */}
                      <GooglePlacesAutocomplete
                        apiKey={googleMapAPIkey}
                        selectProps={{
                          defaultInputValue: `${profile?.address}`,
                          value,
                          isClearable: true,
                          placeholder: `${i18next.t('Search for city/location')}`,
                          onChange: handleChangeLocation,
                        }}
                      />
                    </div>
                  </div>
                  <div className="row align-items-start">
                    <div className="col-12 col-md-3">
                      <h5>Social Profiles</h5>
                    </div>
                    <div className="col-12 col-md-9">
                      <InputField
                        type="text"
                        label="Facebook"
                        error={formState.errors['facebook_url']}
                        registration={register('facebook_url')}
                        placeholder="Facebook"
                      />
                      <InputField
                        type="text"
                        label="Twitter"
                        error={formState.errors['twitter_url']}
                        registration={register('twitter_url')}
                        placeholder="Twitter"
                      />
                      <InputField
                        type="text"
                        label="YouTube"
                        error={formState.errors['youtube_url']}
                        registration={register('youtube_url')}
                        placeholder="You Tube"
                      />
                      <InputField
                        type="text"
                        label="Instagram"
                        error={formState.errors['instagram_url']}
                        registration={register('instagram_url')}
                        placeholder="Instagram"
                      />
                      <InputField
                        type="text"
                        label="Linkedin"
                        error={formState.errors['linkedin_url']}
                        registration={register('linkedin_url')}
                        placeholder="Linkedin"
                      />
                      <InputField
                        type="text"
                        label="Pinterest"
                        error={formState.errors['pintrest_url']}
                        registration={register('pintrest_url')}
                        placeholder="Pinterest"
                      />
                    </div>
                  </div>
                  <div>
                    <Button isLoading={loading} type="submit" className="mx-auto d-inline-block">
                      Update Account
                    </Button>
                  </div>
                </>
              )}
            </Form>
          )}
        </div>
      </div>
    </ContentLayout>
  );
};
