import React, { useState, useEffect } from 'react';
import { GoogleMap, Marker } from '@react-google-maps/api';
import axiosmod from 'axios';
import GooglePlacesAutocomplete from 'react-google-places-autocomplete';
import { ContentLayout } from '@/components/Layout';
import * as z from 'zod';
import { useSelectMultipleFile } from '@/hooks/useSelectMultipleFile';
import { InputPhone } from '@/components/Form/InputPhone';
import { isValidMobile, isValidMobileMsg } from '@/utils/validation';
import plus from '@/assets/plus.png';
import { Form, InputField, SelectField } from '@/components/Form';
import { Button, Spinner } from '@/components/Elements';
import { useNotificationStore } from '@/stores/notifications';
import { useNavigate, useParams } from 'react-router-dom';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { axios } from '@/lib/axios';
import './CreateProperty.css';
import EditDraftInput from '@/components/Elements/DraftInput/EditDraftInput';
import { usePropertyProjectDetails } from '@/api/propertyProject/getPropertyProjectDetails';
import { PropertyProject } from '@/types';
import propertyCategory from '@/constants/propertyCategory';
import amanetiesCategory from '@/constants/amanetiesCategory';
import { MultiSelect } from 'react-multi-select-component';
import { EditorState } from 'draft-js';
import EditPageLoader from '@/components/Elements/Loader/EditPageLoader';
import { useAuth } from '@/lib/auth';
import storage from '@/utils/storage';
import i18next from 'i18next';
import { translateAmenties } from '@/translations/translateFunction';
const googleMapAPIkey = process.env.REACT_APP_GOOGLE_MAPS_API_KEY as string;
// -------------- google maps --------------------

function Map({
  location,
  // addressDetails,
  setAddressDetails,
}: {
  location: any;
  addressDetails: any;
  setAddressDetails: any;
}) {
  const [userLat, setUserLat] = useState(56.130366);
  const [userLong, setUserLong] = useState(-106.346771);
  const center = { lat: userLat, lng: userLong };

  const geocoder = new google.maps.Geocoder();

  useEffect(() => {
    getLocationDetails();
  }, [center]);

  const getLocationDetails = () => {
    geocoder.geocode({ location: center }).then((response) => {
      if (response.results[0]) {
        setAddressDetails({
          latitude: userLat,
          longitude: userLong,
          address_line: response.results[0].formatted_address,
        });
      } else {
        console.log('No results found');
      }
    });
  };

  useEffect(() => {
    if (location?.lat) {
      setUserLat(parseFloat(location?.lat));
      setUserLong(parseFloat(location?.lng));
      getLocationDetails();
    }
  }, [location]);

  return (
    <GoogleMap
      zoom={10}
      center={center}
      mapContainerClassName="map-container"
      onClick={(e) => {
        setUserLat(e.latLng?.lat() || 0);
        setUserLong(e.latLng?.lng() || 0);
      }}
    >
      <Marker position={center} />
    </GoogleMap>
  );
}

// ---------------google maps -------------------

const schema = z.object({
  short_title: z.string().min(1, 'Please enter short title'),
  property_name: z.string().min(1, 'Please enter property name'),
  youtube_url: z.string().min(1, 'Please enter youtube URL'),
  phone_number: z.custom(isValidMobile, isValidMobileMsg),

  price: z
    .string()
    .min(1, 'Please enter price')
    .regex(/^\d+(\.\d+)?$/)
    .transform(Number),
  size: z.string().min(1, 'Please enter size').max(5, 'Property size cannot exceed 5 units'),
  bill_utilites: z.string().min(1, 'Please enter bill utilities'),
  return_on_investment: z
    .string()
    .min(1, 'Please enter return of investments')
    .regex(/^\d+(\.\d+)?$/)
    .transform(Number),
  cashflow_pcm: z
    .string()
    .min(1, 'Please enter cashflow PCM')
    .regex(/^\d+(\.\d+)?$/)
    .transform(Number),
  finder_fees: z
    .string()
    .min(1, 'Please enter finder fees')
    .regex(/^\d+(\.\d+)?$/)
    .transform(Number),
  gross_income: z.string().min(1, 'Please enter gross income'),
  total_closing_fees: z.string().min(1, 'Please enter closing fees'),
  developer: z.string().min(1, 'Please enter purchase developer'),
  bedroom: z.string().min(1, 'Please enter number of bedrooms'),
  bathroom: z.string().min(1, 'Please enter number of bathrooms'),
  read_by: z.string().min(1, 'Please enter Ready by'),
  property_refernce_id: z
    .string()
    .min(1, 'Please enter property reference id')
    .max(10, 'Refrence Id value too large'),
  maintance_fees: z.string().min(1, 'Please enter maintenance fee'),
  house_number: z.string().min(1, 'Please enter house number'),
  // town: z.string().min(1, 'Please enter town'),
  postal_code: z.string().min(1, 'Please enter postal code').max(10, 'Postal code value too large'),
  category: z.string().min(1, 'Please select Property Category'),
  rent_to_landlord: z.string().min(1, 'Please enter rent to landloard'),
});

type CreatePropertyValues = {
  short_title: string;
  category: string;
  images: string;
  property_name: string;
  youtube_url: string;
  phone_number: string;
  disclaimer: string;
  property_information: string;
  price: any;
  size: string;
  bill_utilites: any;
  return_on_investment: any;
  cashflow_pcm: any;
  finder_fees: any;
  gross_income: any;
  total_closing_fees: any;
  developer: string;
  bedroom: string;
  bathroom: string;
  read_by: string;
  property_refernce_id: string;
  maintance_fees: any;
  house_number: string;
  town: string;
  postal_code: string;
  rent_to_landlord: string;
  latitude: string;
  longitude: string;
  is_draft: boolean;
};

type LocationCoordinates = {
  lat?: number;
  lng?: number;
};
export const EditPropertyProject = () => {
  const { id } = useParams();
  const localCurrency = storage.getAmount();
  const localSymbol = storage.getSymbol();
  const navigate = useNavigate();
  const { images, handleChange, uploaded, deleteImage, text, setText, load } =
    useSelectMultipleFile();

  const [loading, setLoading] = useState(false);
  // const [latitude, setlatitude] = useState('');
  // const [longitude, setlongitude] = useState('');
  const { data, isLoading, isRefetching } = usePropertyProjectDetails({ id: id ?? '1' });
  const user = data?.data ?? ({} as PropertyProject);
  const auth = useAuth();
  const [is_draft, setis_draft] = useState<boolean>(false);

  // const handleChangeName = (event) => {
  //   setImageName(event.target.value);
  // };

  // const [imageName, setImageName] = useState('');

  const [imageArray, setImageArray] = useState<any>([]);

  useEffect(() => {
    if (data) {
      const user = data?.data ?? ({} as PropertyProject);
      const previousImage = eval(user?.image_data);
      setImageArray(previousImage);
    }
  }, [data]);

  const deleteImages = (val: any) => {
    const filteredItems = imageArray
      .slice(0, val)
      .concat(imageArray.slice(val + 1, imageArray.length));
    setImageArray(filteredItems);
  };

  const [selected, setSelected] = useState([]);
  const [value, setValue] = useState(null);
  const [latitude, setlatitude] = useState(0);
  const [longitude, setlongitude] = useState(0);
  const [address, setAddress] = useState<string>('');
  const [addressValue, setAddressValue] = useState<string>('');

  const [location, setLocation] = useState<LocationCoordinates>({} as LocationCoordinates);
  const [addressDetails, setAddressDetails] = useState({
    address_label: '',
    latitude: '',
    longitude: '',
  });

  const handleChangeLocation = async (value: any) => {
    const placeId = value.value.place_id;
    const { data } = await axiosmod.get(
      `https://maps.googleapis.com/maps/api/geocode/json?place_id=${placeId}&key=${googleMapAPIkey}`
    );
    const location = data?.results[0]?.geometry?.location;
    if (location) {
      setLocation(location);
      setAddress(value.label);
      setValue(value);
    }
  };

  useEffect(() => {
    setlatitude(addressDetails?.latitude);
    setlongitude(addressDetails?.longitude);
    setAddressValue(addressDetails?.address_line);
  }, [addressDetails]);

  useEffect(() => {
    setlatitude(user?.latitude);
    setlongitude(user?.longitude);
  }, [user]);

  const visionData = user?.vision;
  const rentalData = user?.rental;
  const locationData = user?.location;
  const property_descriptionData = user?.property_description;
  const cashflow_forecastData = user?.cashflow_forecast;
  const costs_to_considerData = user?.costs_to_consider;
  const capital_requiredData = user?.capital_required;
  const disclaimerData = user?.disclaimer;
  const property_informationData = user?.property_information;
  const [visionHtml, setvisionHtml] = useState<any>(null);
  const [locationHtml, setlocationHtml] = useState<any>(null);
  const [rentalHtml, setRentalHtml] = useState<any>(null);
  const [property_descriptionHTML, setproperty_descriptionHTML] = useState<any>(null);
  const [cashflow_forecastHTML, setcashflow_forecastHTML] = useState<any>(null);
  const [costs_to_considerHTML, setcosts_to_considerHTML] = useState<any>(null);
  const [capital_requiredHTML, setcapital_requiredHTML] = useState<any>(null);
  const [disclaimerHTML, setdisclaimerHTML] = useState<any>(null);
  const [property_informationHTML, setproperty_informationHTML] = useState<any>(null);

  const [visionState, setvisionState] = useState(() => EditorState.createEmpty());
  const [locationState, setlocationState] = useState(() => EditorState.createEmpty());
  const [rentalState, setrentalState] = useState(() => EditorState.createEmpty());
  const [property_descriptionState, setproperty_descriptionState] = useState(() =>
    EditorState.createEmpty()
  );
  const [cashflow_forecastState, setcashflow_forecastState] = useState(() =>
    EditorState.createEmpty()
  );
  const [costs_to_considerState, setcosts_to_considerState] = useState(() =>
    EditorState.createEmpty()
  );
  const [capital_requiredState, setcapital_requiredState] = useState(() =>
    EditorState.createEmpty()
  );
  const [disclaimerState, setdisclaimerState] = useState(() => EditorState.createEmpty());
  const [property_informationState, setproperty_informationState] = useState(() =>
    EditorState.createEmpty()
  );

  useEffect(() => {
    if (user && user.amenities) {
      const amenities = user.amenities.split(',');
      const selected = amenities.map((i) => {
        return {
          label: amanetiesCategory[i],
          value: i,
        };
      });
      setSelected(selected);
    }
  }, [user]);

  const handleText = (index: number, val: string) => {
    const textItems = [...text];
    textItems[index] = val;
    setText(textItems);
  };

  const convertValuesToLocalCurrency = async (values: any) => {
    const fieldsToConvert = [
      'return_on_investment',
      'price',
      'finder_fees',
      'cashflow_pcm',
      'bill_utilites',
      'maintance_fees',
      'total_closing_fees',
      'gross_income',
      'rent_to_landlord',
    ];

    fieldsToConvert.forEach((field) => {
      values[field] /= localCurrency;
    });
  };

  const handleSubmit = async (values: CreatePropertyValues) => {
    const newArray = uploaded.map(function (value, index) {
      return { image: value, name: text[index] };
    });

    const image_data = imageArray.concat(newArray);
    if (auth.user?.has_subscription == true) {
      setis_draft(false);
    } else {
      setis_draft(true);
    }
    try {
      setLoading(true);
      const amenities = selected.map(({ label, value }) => `${value}`).join(',');
      const property_id = id;
      const rental = rentalHtml ?? '';
      const vision = visionHtml ?? '';
      const location = locationHtml ?? '';
      const property_description = property_descriptionHTML ?? '';
      const cashflow_forecast = cashflow_forecastHTML ?? '';
      const costs_to_consider = costs_to_considerHTML ?? '';
      const capital_required = capital_requiredHTML ?? '';
      const disclaimer = disclaimerHTML ?? '';
      const property_information = property_informationHTML ?? '';
      const town = addressValue;
      await convertValuesToLocalCurrency(values);
      await axios.post('/customer-api/edit-property-project', {
        ...values,
        property_id,
        image_data,
        is_draft,
        rental,
        disclaimer,
        property_information,
        vision,
        property_description,
        cashflow_forecast,
        costs_to_consider,
        location,
        capital_required,
        latitude,
        town,
        amenities,
        longitude,
      });

      useNotificationStore.getState().addNotification({
        title: 'Success',
        type: 'success',
        message: 'Property submitted successfully!',
      });
      navigate('/user/my-listing');
    } finally {
      setLoading(false);
    }
  };

  return (
    <ContentLayout title="Create Property Project">
      <BackdropLoader open={loading} />
      <div className="container mt-4">
        {isLoading || isRefetching || !localCurrency ? (
          <EditPageLoader />
        ) : (
          <Form<CreatePropertyValues, typeof schema>
            onSubmit={handleSubmit}
            schema={schema}
            options={{
              defaultValues: {
                ...user,
                return_on_investment: (user?.return_on_investment * localCurrency).toString(),
                price: (user?.price * localCurrency).toString(),
                finder_fees: (user?.finder_fees * localCurrency).toString(),
                cashflow_pcm: (user?.cashflow_pcm * localCurrency).toString(),
                bathroom: user?.bathroom?.toString(),
                bedroom: user?.bedroom?.toString(),
                bill_utilites: (user?.bill_utilites * localCurrency).toString(),
                maintance_fees: (user?.maintance_fees * localCurrency).toString(),
                total_closing_fees: (user?.total_closing_fees * localCurrency).toString(),
                gross_income: (user?.gross_income * localCurrency).toString(),
                rent_to_landlord: (user?.rent_to_landlord * localCurrency).toString(),
                // : user?.?.toString(),
              },
            }}
          >
            {({ register, formState }) => (
              <>
                <div className="profile-box mb-5">
                  <div className="row">
                    {imageArray ? (
                      imageArray.map((item: any, index: number) => (
                        <div key={index} className="col-md-2 cross-pic">
                          <div className="image-box">
                            <button type="button" onClick={() => deleteImages(index)}>
                              <i className="fa-solid fa-square-xmark pics"></i>
                            </button>
                            <img src={item.image} alt="..." />
                          </div>
                          {item.name && <h6 className="itemName">{item.name}</h6>}
                        </div>
                      ))
                    ) : (
                      <></>
                    )}

                    {images.map((item: any, index: number) => (
                      <div key={index} className="cross-pic">
                        <div className="image-box">
                          <button type="button" onClick={() => deleteImage(index)}>
                            <i className="fa-solid fa-square-xmark pics"></i>
                          </button>
                          <img src={item} alt="..." />
                        </div>
                        <input
                          type="text"
                          id="imageName"
                          name="imageName"
                          placeholder="Image name"
                          onChange={(e) => handleText(index, e.target.value)}
                          value={text[index]}
                        />
                        {/* {item.name && <h6 className="itemName">{item.name}</h6>} */}
                      </div>
                    ))}

                    <div className="col-md-2 col-4">
                      <div className="padding-r">
                        <label htmlFor="fileInput1">
                          <div className="image-box">
                            {load ? <Spinner /> : <img src={plus} alt="" width="60" />}
                          </div>
                        </label>
                        <input
                          accept="image/*"
                          onChange={handleChange}
                          multiple
                          type="file"
                          id="fileInput1"
                          className="d-none"
                        />
                      </div>
                    </div>
                  </div>

                  <div className="row mt-4">
                    <div>
                      <InputField
                        className="col-6 mt-4"
                        type="text"
                        wrapperClass="w-full"
                        label="Short Title"
                        error={formState.errors['short_title']}
                        registration={register('short_title')}
                      />
                    </div>
                    <div>
                      <SelectField
                        label="Company Category"
                        error={formState.errors['category']}
                        registration={register('category')}
                        placeholder="Company Category"
                        options={Object.entries(propertyCategory).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Property Name"
                        error={formState.errors['property_name']}
                        registration={register('property_name')}
                        placeholder="Property Name"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="float"
                        label="House Number"
                        error={formState.errors['house_number']}
                        registration={register('house_number')}
                        placeholder="House Number"
                      />
                    </div>
                    <div style={{ width: '100%' }}>
                      <div style={{ width: '100%', height: '80%' }}>
                        <Map
                          location={location}
                          setAddressDetails={setAddressDetails}
                          addressDetails={addressDetails}
                        />
                      </div>
                    </div>
                    <br />
                    <div className="col-12">
                      <div className="pt-2 pb-2">
                        <GooglePlacesAutocomplete
                          selectProps={{
                            defaultInputValue: user?.town ? user?.town : '',
                            value,
                            onChange: handleChangeLocation,
                          }}
                        />
                      </div>
                    </div>
                    <div className="col-12 mt-4">
                      <p>{i18next.t('Location')}</p>
                      <EditDraftInput
                        stateData={locationData}
                        html={locationHtml}
                        setHtml={setlocationHtml}
                        editorState={locationState}
                        setEditorState={setlocationState}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="{i18next.t('Postal Code')}"
                        error={formState.errors['postal_code']}
                        registration={register('postal_code')}
                        placeholder="{i18next.t('Postal Code')}"
                      />
                    </div>{' '}
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        min={0}
                        label="Youtube URL"
                        error={formState.errors['youtube_url']}
                        registration={register('youtube_url')}
                        placeholder="Youtube URL"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputPhone
                        value={user?.phone_number}
                        floating
                        error={formState.errors['phone_number']}
                        registration={register('phone_number')}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="float"
                        label={`${i18next.t('Price')} ${localSymbol}`}
                        error={formState.errors['price']}
                        registration={register('price')}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Property dimensions sq.ft"
                        error={formState.errors['size']}
                        registration={register('size')}
                        placeholder="Property dimensions sq.ft"
                      />
                    </div>
                    <div className="col-12 mt-4">
                      <p>{translateAmenties('Vision')}</p>
                      <EditDraftInput
                        stateData={visionData}
                        html={visionHtml}
                        setHtml={setvisionHtml}
                        editorState={visionState}
                        setEditorState={setvisionState}
                      />
                    </div>
                    <div className="col-12 mt-4">
                      <p>{translateAmenties('Rental')}</p>
                      <EditDraftInput
                        stateData={rentalData}
                        html={rentalHtml}
                        setHtml={setRentalHtml}
                        editorState={rentalState}
                        setEditorState={setrentalState}
                      />
                    </div>
                    <div className="col-12 mt-4">
                      <p>{translateAmenties('Property Description')}</p>
                      <EditDraftInput
                        stateData={property_descriptionData}
                        html={property_descriptionHTML}
                        setHtml={setproperty_descriptionHTML}
                        editorState={property_descriptionState}
                        setEditorState={setproperty_descriptionState}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label={`Total closing fees ${localSymbol}`}
                        error={formState.errors['total_closing_fees']}
                        registration={register('total_closing_fees')}
                        placeholder="Total closing fees &#8364;"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Developer"
                        error={formState.errors['developer']}
                        registration={register('developer')}
                        placeholder="Developer"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <SelectField
                        wrapperClass="w-full"
                        label={i18next.t('Bedrooms')}
                        error={formState.errors['bedroom']}
                        registration={register('bedroom')}
                        placeholder="3"
                        options={[
                          { label: 1, value: 1 },
                          { label: 2, value: 2 },
                          { label: 3, value: 3 },
                          { label: 4, value: 4 },
                          { label: 5, value: 5 },
                        ]}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <SelectField
                        wrapperClass="w-full"
                        label="Bathrooms"
                        error={formState.errors['bathroom']}
                        registration={register('bathroom')}
                        placeholder="3"
                        options={[
                          { label: 1, value: 1 },
                          { label: 2, value: 2 },
                          { label: 3, value: 3 },
                          { label: 4, value: 4 },
                          { label: 5, value: 5 },
                        ]}
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Ready by"
                        error={formState.errors['read_by']}
                        registration={register('read_by')}
                        placeholder="Ready by"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Property Refernce ID"
                        error={formState.errors['property_refernce_id']}
                        registration={register('property_refernce_id')}
                        placeholder="Property Refernce ID"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      <InputField
                        type="float"
                        label={`Maintance fees ${localSymbol}`}
                        error={formState.errors['maintance_fees']}
                        registration={register('maintance_fees')}
                        placeholder="Maintance fees &#8364;"
                      />
                    </div>
                    <div className="col-6 mt-4">
                      {/* <p>{amanetiesValue}</p> */}
                      <MultiSelect
                        options={Object.entries(amanetiesCategory).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                        value={selected}
                        onChange={setSelected}
                        labelledBy="Select Amenaties."
                      />
                    </div>
                    <p className="mt-4 btn-heavy">Property Information</p>
                    <div className="col-6">
                      <InputField
                        type="text"
                        label={`Bills & Utilites ${localSymbol}`}
                        error={formState.errors['bill_utilites']}
                        registration={register('bill_utilites')}
                        placeholder="Bills & Utilites &#8364;"
                      />
                    </div>
                    <div className="col-6">
                      <InputField
                        type="text"
                        label={`${i18next.t('Return on investment')} ${localSymbol}`}
                        error={formState.errors['return_on_investment']}
                        registration={register('return_on_investment')}
                        placeholder={`${i18next.t('Return on investment')} &#8364;`}
                      />
                    </div>
                    <div className="col-12 col-md-6 mt-2">
                      <InputField
                        type="text"
                        label={`${i18next.t('Cashflow PCM')} ${localSymbol}`}
                        error={formState.errors['cashflow_pcm']}
                        registration={register('cashflow_pcm')}
                        placeholder="Cashflow PCM &#8364;"
                      />
                    </div>
                    <div className="col-12 col-md-6 mt-2">
                      <InputField
                        type="float"
                        label={`Finder fees ${localSymbol}`}
                        error={formState.errors['finder_fees']}
                        registration={register('finder_fees')}
                        placeholder="Finder fees &#8364;"
                      />
                    </div>
                    <div className="col-12 col-md-6 mt-2">
                      <InputField
                        type="float"
                        label={`Potential gross income ${localSymbol}`}
                        error={formState.errors['gross_income']}
                        registration={register('gross_income')}
                        placeholder="Potential gross income &#8364;"
                      />
                    </div>
                    <div className="col-12 col-md-6 mt-2">
                      <InputField
                        type="float"
                        label={`${i18next.t('Rent To Landlord')} ${localSymbol}`}
                        error={formState.errors['rent_to_landlord']}
                        registration={register('rent_to_landlord')}
                        placeholder={`${i18next.t('Rent To Landlord')} &#8364;`}
                      />
                    </div>
                    <div className="col-12 mt-2">
                      <p>{i18next.t('Cashflow Forecast')}</p>
                      <EditDraftInput
                        stateData={cashflow_forecastData}
                        html={cashflow_forecastHTML}
                        setHtml={setcashflow_forecastHTML}
                        editorState={cashflow_forecastState}
                        setEditorState={setcashflow_forecastState}
                      />
                    </div>
                    <div className="col-12 mt-2">
                      <p>Costs To Consider</p>
                      <EditDraftInput
                        stateData={costs_to_considerData}
                        html={costs_to_considerHTML}
                        setHtml={setcosts_to_considerHTML}
                        editorState={costs_to_considerState}
                        setEditorState={setcosts_to_considerState}
                      />
                    </div>
                    <div className="col-12 mt-2">
                      <p>{i18next.t('Capital Required')}</p>
                      <EditDraftInput
                        stateData={capital_requiredData}
                        html={capital_requiredHTML}
                        setHtml={setcapital_requiredHTML}
                        editorState={capital_requiredState}
                        setEditorState={setcapital_requiredState}
                      />
                    </div>
                    <div className="col-12 mt-2">
                      <p>{i18next.t('Disclaimer')}</p>
                      <EditDraftInput
                        stateData={disclaimerData}
                        html={disclaimerHTML}
                        setHtml={setdisclaimerHTML}
                        editorState={disclaimerState}
                        setEditorState={setdisclaimerState}
                      />
                    </div>
                    <div className="col-12 mt-2">
                      <p>Property Information</p>
                      <EditDraftInput
                        stateData={property_informationData}
                        html={property_informationHTML}
                        setHtml={setproperty_informationHTML}
                        editorState={property_informationState}
                        setEditorState={setproperty_informationState}
                      />
                    </div>
                    <div className="d-flex justify-content-center mt-4">
                      <Button type="submit" block isLoading={loading}>
                        Submit
                      </Button>
                    </div>
                  </div>
                </div>
              </>
            )}
          </Form>
        )}
      </div>
    </ContentLayout>
  );
};
