import React, { useEffect, useState } from 'react';
import * as z from 'zod';
import { Country } from 'country-state-city';

import plus from '@/assets/plus.png';

import { ContentLayout } from '@/components/Layout';
import { useSelectFile } from '@/hooks/useSelectFile';
import { Form, InputField, SelectField, TextAreaField } from '@/components/Form';
import { Button, Spinner } from '@/components/Elements';
import propertyType from '@/constants/propertyType';
import { uploadImage } from '@/api/uploadImage';
import { housingGetId } from '@/api/housing/housingGetId';
import { useNotificationStore } from '@/stores/notifications';
import { useNavigate, useParams } from 'react-router-dom';
import BackdropLoader from '@/components/Elements/Loader/BackdropLoader';
import { useAuth } from '@/lib/auth';
import storage from '@/utils/storage';
import { axios } from '@/lib/axios';
import { Housing } from '@/types';
import EditSalePageLoader from '@/components/Elements/Loader/EditSalePageLoader';
import { useSelectMultipleFile } from '@/hooks/useSelectMultipleFile';
import GooglePlacesAutocomplete from 'react-google-places-autocomplete';
import { GoogleMap, Marker } from '@react-google-maps/api';
import axiosmod from 'axios';
import { InputPhone } from '@/components/Form/InputPhone';
import { isValidMobile, isValidMobileMsg } from '@/utils/validation';
import studyType from '@/constants/studyType';
import partyTypes from '@/constants/partyTypes';
import jobType from '@/constants/jobType';
import cleanType from '@/constants/cleanType';
import i18next from 'i18next';

const googleMapAPIkey = process.env.REACT_APP_GOOGLE_MAPS_API_KEY as string;

// -------------- google maps --------------------

function Map({
  location,
  // addressDetails,
  setAddressDetails,
}: {
  location: any;
  addressDetails: any;
  setAddressDetails: any;
}) {
  const [userLat, setUserLat] = useState(37.0902);
  const [userLong, setUserLong] = useState(95.7129);
  const center = { lat: userLat, lng: userLong };

  const geocoder = new google.maps.Geocoder();

  useEffect(() => {
    getLocationDetails();
  }, [center]);

  const getLocationDetails = () => {
    geocoder.geocode({ location: center }).then((response) => {
      if (response.results[0]) {
        setAddressDetails({
          latitude: userLat,
          longitude: userLong,
          address_line: response.results[0].formatted_address,
        });
      } else {
        console.log('No results found');
      }
    });
  };

  useEffect(() => {
    if (location?.lat) {
      setUserLat(parseFloat(location?.lat));
      setUserLong(parseFloat(location?.lng));
      getLocationDetails();
    }
  }, [location]);

  return (
    <GoogleMap
      zoom={10}
      center={center}
      mapContainerClassName="map-container"
      onClick={(e) => {
        setUserLat(e.latLng?.lat() || 0);
        setUserLong(e.latLng?.lng() || 0);
      }}
    >
      <Marker position={center} />
    </GoogleMap>
  );
}

type LocationCoordinates = {
  lat?: number;
  lng?: number;
};

// ---------------google maps -------------------

const schema = z.object({
  first_name: z.string().min(1, 'Please enter first name'),
  last_name: z.string().min(1, 'Please enter last name'),
  gender: z.string().min(1, 'Please select gender'),
  dob: z.string().min(1, 'Please select date of birth'),
  address: z.string().min(1, 'Please enter address'),
  nationality: z.string().min(1, 'Please select nationality'),
  phone_number: z.custom(isValidMobile, isValidMobileMsg),
  email: z.string().email('Please enter a valid email address').optional(),
  about: z.string().optional(),
  max_month_price: z
    .string()
    .min(1, 'Please enter max month price')
    .regex(/^\d+$/)
    .transform(Number),
  property_type: z.string().min(1, 'Please select property type'),
  wanted_from: z.string().min(1, 'Please enter house wanted from rent date'),
  no_of_resident: z.string().optional(),
  // location: z.string().optional(),
  pet: z.string().optional(),
  furnishings: z.string().optional(),
  ad_header: z.string().optional(),
  cleanliness: z.string().min(1, 'Please enter details'),
  party_habits: z.string().min(1, 'Please enter details'),
  interests: z.string().min(1, 'Please enter details'),
  study_status: z.string().min(1, 'Please enter details'),
  education: z.string().min(1, 'Please enter details'),
  job_status: z.string().min(1, 'Please enter details'),
  work: z.string().min(1, 'Please enter details'),
  linkedIn: z.string().min(1, 'Please enter details'),
});

type HouseRentFormValues = {
  first_name: string;
  last_name: string;
  gender: string;
  dob: string;
  address: string;
  phone_number: string;
  email: string;
  about: string;
  nationality: string;
  max_month_price: string;
  opening_hours: string;
  property_type: string;
  wanted_from: string;
  no_of_resident: string;
  location: string;
  pet: string;
  is_draft: any;
  property_id: any;
  image: any;
  image_data: any;
  ad_header: string;
  furnishings: string;
  latitude: string;
  longitude: string;
  // -----------
  cleanliness: string;
  party_habits: string;
  interests: string;
  study_status: string;
  education: string;
  job_status: string;
  work: string;
  linkedIn: string;
};
const allCountries = Country.getAllCountries().map((i) => ({
  label: i.name,
  value: i.name,
}));

export const HouseRentForm = () => {
  const { file, preview, handleChange } = useSelectFile();
  const [loading, setLoading] = useState(false);
  const localCurrency = storage.getAmount();
  const localSymbol = storage.getSymbol();
  const navigate = useNavigate();

  const auth = useAuth();
  const { id } = useParams();

  // ye h edit k lie

  // const { data, isLoading, isRefetching, refetch } = useHousingIdDetails({ id: id ?? '1' });
  // const user = data?.data ?? ({} as Housing);

  const [user, setUser] = useState<Housing>();
  const [isLoader, setisLoader] = useState<boolean>(false);

  const [value, setValue] = useState(null);
  const [latitude, setlatitude] = useState(0);
  const [longitude, setlongitude] = useState(0);
  const [addressValue, setAddressValue] = useState<string>('');
  const [errorState, setErrorState] = useState<string>('');
  const [showError, setShowError] = useState<boolean>(false);

  const [address, setAddress] = useState<string>('');
  const [location, setLocation] = useState<LocationCoordinates>({} as LocationCoordinates);
  const [addressDetails, setAddressDetails] = useState({
    address_line: '',
    latitude: '',
    longitude: '',
  });

  useEffect(() => {
    setlatitude(addressDetails?.latitude);
    setlongitude(addressDetails?.longitude);
    setAddressValue(addressDetails?.address_line);
  }, [addressDetails]);

  useEffect(() => {
    setlatitude(user?.latitude);
    setlongitude(user?.longitude);
  }, [user]);

  const handleChangeLocation = async (value: any) => {
    const placeId = value.value.place_id;
    const { data } = await axiosmod.get(
      `https://maps.googleapis.com/maps/api/geocode/json?place_id=${placeId}&key=${googleMapAPIkey}`
    );
    const location = data?.results[0]?.geometry?.location;
    if (location) {
      setLocation(location);
      setAddress(value.label);
      setValue(value);
    }
  };

  const getDataOfApi = async () => {
    if (id != null) {
      setisLoader(true);
      const data = await housingGetId(id);
      const userIn = data?.data ?? ({} as Housing);
      setUser(userIn);
      setisLoader(false);
    }
  };
  useEffect(() => {
    auth.refetchUser();
    getDataOfApi();
  }, [id]);

  // ye h edit k lie

  const [idd, setIdd] = useState('');
  const [is_draft, setis_draft] = useState<string>('False');
  const [userSubscribed, setuserSubscribed] = useState();

  const {
    images,
    handleChange: handlechanges,
    uploaded,
    deleteImage,
    text,
    setText,
    load,
  } = useSelectMultipleFile();

  const deleteImages = (val: any) => {
    const filteredItems = imageArray
      .slice(0, val)
      .concat(imageArray.slice(val + 1, imageArray.length));
    setImageArray(filteredItems);
  };

  const [imageArray, setImageArray] = useState<any>([]);
  useEffect(() => {
    if (user) {
      const previousImage = eval(user?.image_data);
      setImageArray(previousImage);
    }
  }, [user]);

  const handleText = (index: number, val: string) => {
    const textItems = [...text];
    textItems[index] = val;
    setText(textItems);
  };

  useEffect(() => {
    auth.refetchUser();
    setuserSubscribed(auth.user?.has_subscription);
  }, [auth.user.has_subscription]);

  useEffect(() => {
    if (auth.user?.has_subscription == true) {
      setis_draft('False');
    } else {
      setis_draft('True');
    }
  }, [auth?.user?.has_subscription]);

  const addData = {
    editId: `${idd}`,
    pageUrl: `/user/create-house-rent/${idd}?type=SaleProperty`,
  };
  storage.setSubmitFormData(addData);

  const showSuccess = () => {
    useNotificationStore.getState().addNotification({
      title: 'Success',
      type: 'success',
      message: 'Property Listing SUbmitted Successfully!',
    });
  };

  const handleSubmit = async (values: HouseRentFormValues) => {
    try {
      setLoading(true);
      let imageUrl = '';

      if (file) {
        const response = await uploadImage({ file });
        imageUrl = response.url;
      } else {
        imageUrl = user?.image ?? '';
      }

      const x = imageArray?.length ? imageArray?.length : uploaded?.length;

      if (x >= 1) {
        setShowError(false);
        const newArray = uploaded.map(function (value, index) {
          return { image: value, name: text[index] };
        });

        const imageData = imageArray.concat(newArray);
        values.max_month_price /= localCurrency;
        values.property_id = id ?? '1';
        values.is_draft = is_draft;
        values.image = imageUrl;
        values.image_data = imageData;
        values.latitude = latitude;
        values.longitude = longitude;
        values.location = addressValue;

        if (id != null) {
          axios
            .post('/customer-api/edit-house-wanted', values)
            .then(() => {
              setLoading(false);
              navigate('/user/my-listing');
              showSuccess();
            })
            .catch(() => {
              setLoading(false);
            });
        } else {
          axios
            .post('/customer-api/house-wanted-for-rent', values)
            .then((response) => {
              setLoading(false);
              showSuccess();
              if (auth.user?.has_subscription == true) {
                setLoading(false);
                navigate('/housing-for-rent');
              } else {
                setIdd(response?.data?.id);
                navigate('/user/subscription');
                setLoading(false);
              }
            })
            .catch(() => {
              setLoading(false);
            });
        }
      } else {
        window.scroll(0, 0);
        setShowError(true);
        setErrorState('Image count must be greater than or equal to 1.');
        setLoading(false);
      }
    } finally {
      setLoading(false);
    }
  };
  const [canAddMore, setcanAddMore] = useState(false);
  useEffect(() => {
    if (auth.user?.total_property_upload >= auth.user?.property_count) {
      setcanAddMore(true);
    }
  }, [auth?.user?.has_subscription]);

  return (
    <ContentLayout title="Create House for Rent">
      <button onClick={() => navigate(-1)}>
        <i className="fa-regular fa-circle-left go-back-button"></i>
      </button>
      <BackdropLoader open={loading} />

      <div className="container cret-house-rent mt-4">
        <h3>Create Profile for Tenant</h3>
        <h5 className="ms-2 my-4">Personal Information</h5>

        {isLoader ? (
          <EditSalePageLoader />
        ) : (
          <>
            <Form<HouseRentFormValues, typeof schema>
              onSubmit={handleSubmit}
              schema={schema}
              options={{
                defaultValues: {
                  ...user,
                  max_month_price: user?.max_month_price
                    ? (user?.max_month_price * localCurrency)?.toString()
                    : '',
                },
              }}
            >
              {({ register, formState }) => (
                <div className="profile-box mb-5">
                  <div className="row">
                    <div className="col-12 col-md-12">
                      <div className="d-flex">
                        <div>
                          <h5>Personal Profile</h5>
                          <label htmlFor="fileInput">
                            <div className="image-box">
                              {preview ? (
                                <img src={preview} alt="" className="profileImage" />
                              ) : user?.image ? (
                                <img src={user?.image} alt="" width="60" />
                              ) : (
                                <img src={plus} alt="" width="60" />
                              )}
                            </div>
                          </label>
                          <input
                            accept="image/*"
                            onChange={handleChange}
                            type="file"
                            id="fileInput"
                            className="d-none"
                          />
                        </div>
                      </div>
                    </div>
                    <div className="col-12 col-md-12">
                      <div className="selectedImagesDiv">
                        {showError ? <p>{errorState}</p> : null}
                        <h5>Images</h5>

                        <div className="flx-tenants d-flex">
                          {imageArray.map((item: any, index: number) => (
                            <div key={index} className="col-md-2 cross-pic">
                              <div className="image-box">
                                <button type="button" onClick={() => deleteImages(index)}>
                                  <i className="fa-solid fa-square-xmark pics"></i>
                                </button>
                                <img src={item.image} alt="..." />
                              </div>
                              {item.name && <h6 className="itemName">{item.name}</h6>}
                            </div>
                          ))}

                          {images.map((item: any, index: number) => (
                            <div key={index} className="cross-pic">
                              <div className="image-box">
                                <button type="button" onClick={() => deleteImage(index)}>
                                  <i className="fa-solid fa-square-xmark pics"></i>
                                </button>
                                <img src={item} alt="..." />
                              </div>
                              <input
                                type="text"
                                id="imageName"
                                name="imageName"
                                placeholder="Image name"
                                onChange={(e) => handleText(index, e.target.value)}
                                value={text[index]}
                              />
                            </div>
                          ))}

                          <div className="col-md-2 col-12">
                            <div className="padding-r">
                              <label htmlFor="fileInput1">
                                <div className="image-box">
                                  {load ? <Spinner /> : <img src={plus} alt="" width="60" />}
                                </div>
                              </label>
                              <input
                                accept="image/*"
                                onChange={handlechanges}
                                multiple
                                type="file"
                                id="fileInput1"
                                className="d-none"
                              />
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>

                  <div className="row mt-4">
                    <div className="col-12 col-md-6">
                      <InputField
                        type="text"
                        label={i18next.t('First name')}
                        error={formState.errors['first_name']}
                        registration={register('first_name')}
                        placeholder={i18next.t('First name')}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="text"
                        label={i18next.t('Last name')}
                        error={formState.errors['last_name']}
                        registration={register('last_name')}
                        placeholder={i18next.t('Last name')}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <SelectField
                        label="Gender"
                        error={formState.errors['gender']}
                        registration={register('gender')}
                        placeholder="Gender"
                        options={['Male', 'Female'].map((o) => ({
                          label: o,
                          value: o.toLowerCase(),
                        }))}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="text"
                        label="Address"
                        error={formState.errors['address']}
                        registration={register('address')}
                        placeholder="Address"
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="date"
                        label="Date of birth"
                        error={formState.errors['dob']}
                        registration={register('dob')}
                        placeholder="Date of birth"
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <SelectField
                        label="Nationality*"
                        options={allCountries}
                        error={formState.errors['nationality']}
                        registration={register('nationality')}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputPhone
                        floating
                        error={formState.errors['phone_number']}
                        registration={register('phone_number')}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="email"
                        label={i18next.t('Email')}
                        error={formState.errors['email']}
                        registration={register('email')}
                      />
                    </div>

                    <div className="col-12 col-md-6">
                      <InputField
                        type="text"
                        label={i18next.t('Interests')}
                        error={formState.errors['interests']}
                        registration={register('interests')}
                        placeholder=""
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="text"
                        label={i18next.t('Education')}
                        error={formState.errors['education']}
                        registration={register('education')}
                        placeholder=""
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="text"
                        label={i18next.t('Work')}
                        error={formState.errors['work']}
                        registration={register('work')}
                        placeholder=""
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="text"
                        label={i18next.t('LinkedIn')}
                        error={formState.errors['linkedIn']}
                        registration={register('linkedIn')}
                        placeholder=""
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <SelectField
                        label={i18next.t('Study Status')}
                        error={formState.errors['study_status']}
                        registration={register('study_status')}
                        placeholder="Field"
                        options={Object.entries(studyType).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <SelectField
                        label={i18next.t('Party Habits')}
                        error={formState.errors['party_habits']}
                        registration={register('party_habits')}
                        placeholder="Field"
                        options={Object.entries(partyTypes).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <SelectField
                        label={i18next.t('Job Status')}
                        error={formState.errors['job_status']}
                        registration={register('job_status')}
                        placeholder="Field"
                        options={Object.entries(jobType).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <SelectField
                        label={i18next.t('Cleanliness')}
                        error={formState.errors['cleanliness']}
                        registration={register('cleanliness')}
                        placeholder="Field"
                        options={Object.entries(cleanType).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                      />
                    </div>
                    <div className="col-12 col-md-12">
                      <TextAreaField
                        rows={5}
                        label="Tell About Your Self"
                        error={formState.errors['about']}
                        registration={register('about')}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="float"
                        label={`Max Monthly Price ${localSymbol}`}
                        error={formState.errors['max_month_price']}
                        registration={register('max_month_price')}
                        placeholder="Max Monthly Price (In Dollars)"
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <SelectField
                        label={i18next.t('Property Type')}
                        error={formState.errors['property_type']}
                        registration={register('property_type')}
                        placeholder="Field"
                        options={Object.entries(propertyType).map(([key, value]) => ({
                          label: value,
                          value: key,
                        }))}
                      />
                    </div>

                    <div className="col-12 col-md-6">
                      <InputField
                        type="text"
                        label="Add Short header"
                        error={formState.errors['ad_header']}
                        registration={register('ad_header')}
                        placeholder="I am john ,  i want house"
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="text"
                        label="Furnishings "
                        error={formState.errors['furnishings']}
                        registration={register('furnishings')}
                        placeholder="Furnishings "
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="date"
                        label={i18next.t('Wanted From')}
                        error={formState.errors['wanted_from']}
                        registration={register('wanted_from')}
                        placeholder={i18next.t('Wanted From')}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <InputField
                        type="number"
                        label="Number of residents"
                        error={formState.errors['no_of_resident']}
                        registration={register('no_of_resident')}
                        placeholder="Number of residents"
                      />
                    </div>
                    {/* <div className="col-6 mt-4">
                      <InputField
                        type="text"
                        label="Desired Location"
                        error={formState.errors['location']}
                        registration={register('location')}
                        placeholder="Desired Location"
                      />
                    </div> */}
                    <div className="col-12 col-md-6">
                      <SelectField
                        label="Pet or Animal"
                        error={formState.errors['pet']}
                        registration={register('pet')}
                        placeholder="Pet or Animal"
                        options={['Yes', 'No'].map((key) => ({
                          label: key,
                          value: key,
                        }))}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <div className="pb-2">
                        <GooglePlacesAutocomplete
                          selectProps={{
                            defaultInputValue: user?.location ? user?.location : '',
                            value,
                            onChange: handleChangeLocation,
                          }}
                        />
                      </div>
                    </div>
                    <div style={{ width: '100%' }}>
                      <div style={{ width: '100%', height: '80%' }}>
                        <Map
                          location={location}
                          setAddressDetails={setAddressDetails}
                          addressDetails={addressDetails}
                        />
                      </div>
                    </div>
                    <br />

                    <div className="d-flex justify-content-center mt-4">
                      {userSubscribed == true ? (
                        <>
                          {canAddMore ? (
                            <Button disabled className="disableSubmit">
                              Reached Upload limit
                            </Button>
                          ) : (
                            <Button
                              isLoading={loading}
                              type="submit"
                              variant="primary"
                              size="md"
                              className="mt-2 "
                            >
                              Submit
                            </Button>
                          )}
                        </>
                      ) : (
                        <>
                          {' '}
                          <Button
                            isLoading={loading}
                            type="submit"
                            variant="primary"
                            size="md"
                            className="mt-2"
                          >
                            Subscribe
                          </Button>
                        </>
                      )}
                    </div>
                    {/* <div className="d-flex justify-content-end mt-4">
                  <Button isLoading={loading} type="submit" block>
                    Submit
                  </Button>
                </div> */}
                  </div>
                </div>
              )}
            </Form>
          </>
        )}
      </div>
    </ContentLayout>
  );
};
