import os
import json
from django.http import JsonResponse
from django.views.decorators.csrf import csrf_exempt

from langchain_community.document_loaders import PyPDFLoader, CSVLoader, UnstructuredExcelLoader
from langchain.text_splitter import RecursiveCharacterTextSplitter
from langchain_community.vectorstores import FAISS
from langchain_community.chat_models import ChatOpenAI
from langchain.memory import ConversationBufferMemory
from langchain.chains import ConversationalRetrievalChain
from langchain_huggingface import HuggingFaceEmbeddings

from dotenv import load_dotenv

from user_panel.models import ChatBot
from .models import Conversation

load_dotenv()

embeddings = HuggingFaceEmbeddings(model_name="sentence-transformers/all-MiniLM-L6-v2")

VECTORSTORES = {}
CONVERSATIONS = {}

GREETINGS = {
    "hi": "Hello! How can I help you today?",
    "hello": "Hi there! How are you doing?",
    "hey": "Hey! What can I do for you?",
    "how are you": "I’m just a bot, but I’m doing great! 😊",
    "bye": "Goodbye! Take care!",
    "thank you": "You're welcome!",
    "thanks": "Glad I could help!"
}

INTENTS = {
    "order": "You can place your order through our platform or contact support.",
    "menu": "Would you like to see the menu?",
    "book": "Booking details are not in the uploaded files, please contact reception.",
    "recommend": "Pasta and desserts are highly recommended!"
}


def load_documents(user_id, chatbot_name):
    base_path = os.path.join("media", "data", str(user_id), chatbot_name)
    if not os.path.exists(base_path):
        return []

    documents = []
    for filename in os.listdir(base_path):
        full_path = os.path.join(base_path, filename)
        if filename.endswith(".pdf"):
            loader = PyPDFLoader(full_path)
        elif filename.endswith(".csv"):
            loader = CSVLoader(full_path)
        elif filename.endswith(".xlsx"):
            loader = UnstructuredExcelLoader(full_path)
        else:
            continue
        documents.extend(loader.load())
    return documents


def build_or_load_vectorstore(user_id, chatbot_name, load_existing=False):
    vectorstore_path = os.path.join("vectorstores", f"{user_id}_{chatbot_name}")
    os.makedirs("vectorstores", exist_ok=True)

    if load_existing and os.path.exists(vectorstore_path):
        return FAISS.load_local(vectorstore_path, embeddings, allow_dangerous_deserialization=True)

    docs = load_documents(user_id, chatbot_name)
    if not docs:
        return None

    splitter = RecursiveCharacterTextSplitter(chunk_size=1000, chunk_overlap=200)
    split_docs = splitter.split_documents(docs)

    vectorstore = FAISS.from_documents(split_docs, embeddings)
    vectorstore.save_local(vectorstore_path)
    return vectorstore


def get_or_create_qa(user_id, chatbot_name, load_existing=False):
    key = f"{user_id}_{chatbot_name}"

    if key not in VECTORSTORES:
        vs = build_or_load_vectorstore(user_id, chatbot_name, load_existing)
        if not vs:
            return None, None
        VECTORSTORES[key] = vs
    else:
        vs = VECTORSTORES[key]

    retriever = vs.as_retriever(search_kwargs={"k": 5})
    llm = ChatOpenAI(model="gpt-4o-mini", temperature=0.3)
    memory = ConversationBufferMemory(memory_key="chat_history", return_messages=True)

    qa = ConversationalRetrievalChain.from_llm(llm, retriever=retriever, memory=memory)
    return qa, memory


# def save_lead_from_request(data):
#     required_fields = ["name", "email", "number", "inquiry_type"]
#     if not all(field in data for field in required_fields):
#         return {"error": "Missing one or more required lead fields."}

#     lead_data = {
#         "name": data["name"],
#         "email": data["email"],
#         "number": data["number"],
#         "inquiry_type": data["inquiry_type"],
#     }

#     os.makedirs("data", exist_ok=True)
#     lead_file = "data/leads.json"

#     if os.path.exists(lead_file):
#         with open(lead_file, "r") as f:
#             leads = json.load(f)
#     else:
#         leads = []

#     leads.append(lead_data)

#     with open(lead_file, "w") as f:
#         json.dump(leads, f, indent=4)

#     return {"success": True, "message": "📝 Lead saved successfully!"}





@csrf_exempt
def chat_with_bot(request, chatbot_id):
    if request.method != "POST":
        return JsonResponse({"error": "POST method required"}, status=400)

    try:
        try:
            data = json.loads(request.body)
        except json.JSONDecodeError:
            return JsonResponse({"error": "Invalid JSON body."}, status=400)

        query = data.get("query", "").strip()
        load_existing = data.get("load_existing", True)

        chatbot = ChatBot.objects.filter(id=chatbot_id).first()
        if not chatbot:
            return JsonResponse({"error": "Chatbot not found for this user."}, status=404)

        user_id = chatbot.user_id
        if not user_id or not query:
            return JsonResponse({"error": "Query is required in the request body."}, status=400)

        # Fetch chatbot by id and user_id to confirm access
        chatbot = ChatBot.objects.filter(id=chatbot_id, user_id=user_id).first()
        if not chatbot:
            return JsonResponse({"error": "Chatbot not found for this user."}, status=404)

        chatbot_name = chatbot.chatbot_name
        key = f"{user_id}_{chatbot_name}"

        global CONVERSATIONS

        if key not in CONVERSATIONS:
            qa, memory = get_or_create_qa(user_id, chatbot_name, load_existing)
            if not qa:
                return JsonResponse({"error": "No documents found for this chatbot."}, status=500)
            CONVERSATIONS[key] = {"qa": qa, "memory": memory, "chat_history": []}
        else:
            qa = CONVERSATIONS[key]["qa"]

        normalized_query = query.lower()

        if normalized_query in GREETINGS:
            response = GREETINGS[normalized_query]
        else:
            intent_match = next((v for k, v in INTENTS.items() if k in normalized_query), None)
            if intent_match:
                response = intent_match
            else:
                response = qa.run(query)
                if response.lower() in ["i don't know", "i am not sure"]:
                    response = "⚠️ Sorry, I could not find that information in your documents."

        CONVERSATIONS[key]["chat_history"].append({"user": query, "bot": response})

        chat_history = CONVERSATIONS[key]["chat_history"]
        conversation_obj, created = Conversation.objects.get_or_create(
            user_id=user_id,
            domain=chatbot_name,
            defaults={"chat_history": chat_history}
        )
        if not created:
            conversation_obj.chat_history = chat_history
            conversation_obj.save()

        return JsonResponse({"response": response})

    except Exception as e:
        return JsonResponse({"error": f"Unexpected error: {str(e)}"}, status=500)

@csrf_exempt
def chat_with_bot(request, chatbot_id):
    if request.method != "POST":
        return JsonResponse({"error": "POST method required"}, status=400)

    try:
        try:
            data = json.loads(request.body)
        except json.JSONDecodeError:
            return JsonResponse({"error": "Invalid JSON body."}, status=400)

        query = data.get("query", "").strip()
        load_existing = data.get("load_existing", True)

        chatbot = ChatBot.objects.filter(id=chatbot_id).first()
        if not chatbot:
            return JsonResponse({"error": "Chatbot not found for this user."}, status=404)

        user_id = chatbot.user_id
        if not user_id or not query:
            return JsonResponse({"error": "Query is required in the request body."}, status=400)

        chatbot_name = chatbot.chatbot_name
        key = f"{user_id}_{chatbot_name}"

        global CONVERSATIONS

        if key not in CONVERSATIONS:
            qa, memory = get_or_create_qa(user_id, chatbot_name, load_existing)
            if not qa:
                return JsonResponse({"error": "No documents found for this chatbot."}, status=500)
            CONVERSATIONS[key] = {"qa": qa, "memory": memory, "chat_history": []}
        else:
            qa = CONVERSATIONS[key]["qa"]

        normalized_query = query.lower()

        if normalized_query in GREETINGS:
            response = GREETINGS[normalized_query]
        else:
            intent_match = next((v for k, v in INTENTS.items() if k in normalized_query), None)
            if intent_match:
                response = intent_match
            else:
                response = qa.run(query)
                if response.lower() in ["i don't know", "i am not sure"]:
                    response = "⚠️ Sorry, I could not find that information in your documents."

        # else:
        #     intent_match = next((k for k in ["book", "order"] if k in normalized_query), None)

        #     if intent_match:
        #         # Prompt client to provide lead info
        #         response = (
        #             "Sure! To help you with that, please provide your name, email, phone number, and the type of inquiry."
        #         )
        #     else:
        #         response = qa.run(query)
        #         if response.lower() in ["i don't know", "i am not sure"]:
        #             response = "⚠️ Sorry, I could not find that information in your documents."

        CONVERSATIONS[key]["chat_history"].append({"user": query, "bot": response})

        chat_history = CONVERSATIONS[key]["chat_history"]
        conversation_obj, created = Conversation.objects.get_or_create(
            user_id=user_id,
            domain=chatbot_name,
            defaults={"chat_history": chat_history}
        )
        if not created:
            conversation_obj.chat_history = chat_history
            conversation_obj.save()

        return JsonResponse({"response": response})

    except Exception as e:
        return JsonResponse({"error": f"Unexpected error: {str(e)}"}, status=500)
