from __future__ import annotations

import argparse
from django.core.management.base import BaseCommand
from chatcore.models import FAQ, Document
from chatcore.rag import IngestItem, upsert_items


class Command(BaseCommand):
    help = "Ingest FAQs and Documents into the vector store"

    def add_arguments(self, parser: argparse.ArgumentParser) -> None:
        parser.add_argument("--only", choices=["faqs", "docs", "all"], default="all")

    def handle(self, *args, **options):
        only = options["only"]
        items: list[IngestItem] = []

        if only in ("all", "faqs"):
            for faq in FAQ.objects.all():
                items.append(
                    IngestItem(
                        id=f"faq-{faq.id}",
                        title=faq.question,
                        url=faq.source_url or "",
                        content=faq.answer,
                        snippet=faq.answer[:200],
                    )
                )

        if only in ("all", "docs"):
            for doc in Document.objects.all():
                items.append(
                    IngestItem(
                        id=f"doc-{doc.id}",
                        title=doc.title,
                        url=doc.url,
                        content=doc.content,
                        snippet=(doc.content[:200] if doc.content else ""),
                    )
                )

        count = upsert_items(items)
        self.stdout.write(self.style.SUCCESS(f"Upserted {count} chunks."))


