from __future__ import annotations

import argparse
from django.core.management.base import BaseCommand
from django.db import connection
from chatcore.rag import IngestItem, upsert_items


class Command(BaseCommand):
    help = "Ingest WordPress content (posts/pages/FAQs) from the configured DB into FAISS"

    def add_arguments(self, parser: argparse.ArgumentParser) -> None:
        parser.add_argument("--limit", type=int, default=500)
        parser.add_argument("--post_types", nargs="*", default=["post", "page"])  # add custom types as needed

    def handle(self, *args, **options):
        limit = options["limit"]
        post_types = options["post_types"]

        with connection.cursor() as cur:
            in_clause = ",".join(["%s"] * len(post_types))
            sql = (
                f"SELECT ID, post_title, post_content, post_type, guid "
                f"FROM wp_posts WHERE post_status='publish' AND post_type IN ({in_clause}) "
                f"ORDER BY ID DESC LIMIT %s"
            )
            cur.execute(sql, [*post_types, limit])
            rows = cur.fetchall()

        items: list[IngestItem] = []
        for row in rows:
            pid, title, content, post_type, guid = row
            if not content:
                continue
            items.append(
                IngestItem(
                    id=f"wp-{post_type}-{pid}",
                    title=title or post_type,
                    url=guid or "",
                    content=content,
                    snippet=(content[:200] if content else ""),
                )
            )

        count = upsert_items(items)
        self.stdout.write(self.style.SUCCESS(f"Indexed {len(items)} posts/pages into {count} chunks."))


