"""
Django management command to toggle emoji setting in WordPress database
"""
from django.core.management.base import BaseCommand
from django.db import connection
from chatcore.wordpress_utils import should_use_emojis

class Command(BaseCommand):
    help = 'Toggle emoji setting in WordPress database'

    def add_arguments(self, parser):
        parser.add_argument(
            '--enable',
            action='store_true',
            help='Enable emojis (set to 1)',
        )
        parser.add_argument(
            '--disable',
            action='store_true',
            help='Disable emojis (set to 0)',
        )
        parser.add_argument(
            '--status',
            action='store_true',
            help='Show current emoji status',
        )

    def handle(self, *args, **options):
        if options['status']:
            current_status = should_use_emojis()
            self.stdout.write(
                self.style.SUCCESS(f'Current emoji status: {"ENABLED" if current_status else "DISABLED"}')
            )
            return

        if options['enable']:
            self.set_emoji_setting(1, 'ENABLED')
        elif options['disable']:
            self.set_emoji_setting(0, 'DISABLED')
        else:
            self.stdout.write(
                self.style.ERROR('Please specify --enable, --disable, or --status')
            )

    def set_emoji_setting(self, value, status_text):
        try:
            with connection.cursor() as cursor:
                # Check if the option exists
                cursor.execute(
                    "SELECT option_id FROM joxd_options WHERE option_name = %s",
                    ['acp_enable_emoji']
                )
                result = cursor.fetchone()
                
                if result:
                    # Update existing option
                    cursor.execute(
                        "UPDATE joxd_options SET option_value = %s WHERE option_name = %s",
                        [str(value), 'acp_enable_emoji']
                    )
                    self.stdout.write(
                        self.style.SUCCESS(f'Updated emoji setting to {status_text}')
                    )
                else:
                    # Insert new option
                    cursor.execute(
                        "INSERT INTO joxd_options (option_name, option_value, autoload) VALUES (%s, %s, %s)",
                        ['acp_enable_emoji', str(value), 'yes']
                    )
                    self.stdout.write(
                        self.style.SUCCESS(f'Created emoji setting and set to {status_text}')
                    )
                    
        except Exception as e:
            self.stdout.write(
                self.style.ERROR(f'Error updating emoji setting: {str(e)}')
            )


