from django.conf import settings
from rest_framework.decorators import api_view
from rest_framework.response import Response
from rest_framework import status
from openai import OpenAI
from .rag import search as rag_search
from .wordpress_utils import get_chatbot_settings, get_chatbot_tone, should_use_emojis
import re
import os
import time


def search_knowledge(query: str, limit: int = 5):
    return rag_search(query, top_k=limit)


def openai_fallback(prompt: str, tone: str = "informative", emojis: bool = True, preface_no_site: bool = False):
    api_key = getattr(settings, 'OPENAI_API_KEY', '') or os.getenv('OPENAI_API_KEY', '')
    if not api_key:
        return "OpenAI API key not configured."
    client = OpenAI(api_key=api_key)
    style = f"Tone: {tone}. {'Use emojis.' if emojis else 'Do not use emojis.'}"
    preface = "First acknowledge that no site content was found, then answer succinctly. " if preface_no_site else ""
    sys = (
        "You are a helpful assistant for Alpaca Eco Farm. "
        + preface
        + style
    )
    msg = [
        {"role": "system", "content": sys},
        {"role": "user", "content": prompt},
    ]
    try:
        resp = client.chat.completions.create(
            model=getattr(settings, 'OPENAI_MODEL', 'gpt-4o-mini'),
            messages=msg,
            temperature=getattr(settings, 'OPENAI_TEMPERATURE', 0.2),
            max_tokens=getattr(settings, 'OPENAI_MAX_TOKENS', 300),
        )
        return resp.choices[0].message.content
    except Exception as e:
        return f"AI error: {e}"


@api_view(["POST"]) 
def chat(request):
    data = request.data or {}
    query = data.get("message", "").strip()
    if not query:
        return Response({"error": "message is required"}, status=status.HTTP_400_BAD_REQUEST)
    
    # Get dynamic settings from WordPress database
    wp_settings = get_chatbot_settings()
    tone = get_chatbot_tone()
    emojis = should_use_emojis()

    # 1) Search knowledge base (RAG)
    max_results = getattr(settings, 'CHAT_MAX_RESULTS', 15)
    hits = search_knowledge(query, limit=max_results)
    if hits:
        # Quick keyword-aware extraction for common intents (price)
        lowered = query.lower()
        if "price" in lowered or "cost" in lowered:
            # Try direct regex over snippets first for robustness
            price_candidates = []
            for h in hits:
                snip = h.get("snippet") or ""
                # Match patterns like: price: 15, price 15, $15, 15 EUR, 15.00
                for m in re.findall(r"(?:price\s*[:\-]?\s*|\$)\s*([0-9]+(?:\.[0-9]{1,2})?)", snip, flags=re.I):
                    try:
                        price_candidates.append(float(m))
                    except Exception:
                        pass
            if price_candidates:
                best = min(price_candidates)
                return Response({
                    "source": "site",
                    "message": f"The price I found is {best:.2f} (from site data).",
                    "results": hits[:3],
                })

        # Extract name and email from various patterns
        name = None
        email_query = None
        
        # First, check for email patterns (priority over name extraction)
        m_email = re.search(r"[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,}", query)
        if m_email:
            email_query = m_email.group(0).lower()
        # Also try patterns like "for neha@fiverr.com", "of neha@fiverr.com"
        if not email_query:
            m_email2 = re.search(r"(?:for|of|by)\s+([A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,})", query, re.I)
            if m_email2:
                email_query = m_email2.group(1).lower()
        
        # Only extract name if no email was found
        if not email_query:
            # Extract name patterns: "email of neha", "find user neha", "bookings for neha", "payments for neha"
            m_name = re.search(r"(?:email\s+(?:of|for)|find\s+user|user|bookings?\s+(?:for|of)|payments?\s+(?:for|of))\s+([a-zA-Z]+)(?:\s+([a-zA-Z]+))?", lowered)
            if not m_name:
                m_name = re.search(r"([a-zA-Z]+)\s+(?:email|mail)", lowered)
            if m_name:
                first = m_name.group(1)
                last = m_name.group(2) if (m_name.lastindex and m_name.lastindex >= 2) else None
                name = (first + (f" {last}" if last else "")).strip()

        # Handle specific query types: bookings, payments, or general user lookup
        if name or email_query:
            matched = []
            customer_id = None
            booking_ids = []
            
            # First pass: find user record to get customer ID
            for h in hits:
                snip_lower = (h.get("snippet") or "").lower()
                if email_query and (f"emaillower: {email_query}" in snip_lower or f"email: {email_query}" in snip_lower or email_query in snip_lower):
                    # Extract customer ID from user record
                    snip = h.get("snippet") or ""
                    for m in re.findall(r"id:\s*(\d+)", snip):
                        try:
                            customer_id = int(m)
                            break
                        except:
                            pass
                    matched.append(h)
                    break
            
            # Second pass: find all related records (bookings, payments) for this customer
            for h in hits:
                snip_lower = (h.get("snippet") or "").lower()
                snip = h.get("snippet") or ""
                ok = False
                
                # Direct name matching
                if name:
                    tokens = name.split()
                    ok = (
                        (f"fullname: {name.lower()}" in snip_lower)
                        or (f"fullnamelower: {name.lower()}" in snip_lower)
                        or (len(tokens) > 0 and f"firstname: {tokens[0]}" in snip_lower)
                        or (len(tokens) > 1 and f"lastname: {tokens[1]}" in snip_lower)
                        or (len(tokens) > 0 and f"firstnamelower: {tokens[0]}" in snip_lower)
                        or (len(tokens) > 1 and f"lastnamelower: {tokens[1]}" in snip_lower)
                    )
                
                # Email matching
                if email_query and (f"emaillower: {email_query}" in snip_lower or f"email: {email_query}" in snip_lower or email_query in snip_lower):
                    ok = True
                
                # Customer ID matching (for bookings)
                if customer_id and f"customerid: {customer_id}" in snip_lower:
                    ok = True
                    # Also collect booking IDs for payment matching
                    for m in re.findall(r"id:\s*(\d+)", snip):
                        try:
                            booking_ids.append(int(m))
                        except:
                            pass
                
                # Booking ID matching (for payments)
                if booking_ids:
                    for bid in booking_ids:
                        if f"customerbookingid: {bid}" in snip_lower:
                            ok = True
                            break
                
                if ok:
                    matched.append(h)

            if matched:
                # Check if this is a bookings query
                if ("booking" in lowered or "bookings" in lowered):
                    # Filter only booking records
                    booking_records = [h for h in matched if "booking" in h.get("title", "").lower()]
                    if not booking_records:
                        # If no direct booking records, look for them in all hits
                        for h in hits:
                            if "booking" in h.get("title", "").lower():
                                snip = h.get("snippet") or ""
                                if customer_id and f"customerid: {customer_id}" in snip.lower():
                                    booking_records.append(h)
                    
                    if not booking_records:
                        who = name or email_query
                        return Response({
                            "source": "site",
                            "message": f"No bookings found for {who}.",
                            "results": [],
                        })
                    
                    # Create detailed booking list
                    booking_details = []
                    total_price = 0.0
                    persons_total = 0
                    
                    for h in booking_records:
                        snip = h.get("snippet") or ""
                        booking_id = ""
                        price = 0.0
                        status = ""
                        persons = 0
                        token = ""
                        
                        # Extract booking details
                        for m in re.findall(r"id:\s*(\d+)", snip):
                            booking_id = m
                            break
                        for mp in re.findall(r"price:\s*([0-9]+(?:\.[0-9]{1,2})?)", snip, flags=re.I):
                            try:
                                price = float(mp)
                                total_price += price
                            except Exception:
                                pass
                        for ms in re.findall(r"status:\s*([^\n]+)", snip, flags=re.I):
                            status = ms.strip()
                        for mper in re.findall(r"persons:\s*(\d+)", snip, flags=re.I):
                            try:
                                persons = int(mper)
                                persons_total += persons
                            except Exception:
                                pass
                        for mt in re.findall(r"token:\s*([^\n]+)", snip, flags=re.I):
                            token = mt.strip()
                        
                        booking_details.append({
                            "id": booking_id,
                            "price": price,
                            "status": status,
                            "persons": persons,
                            "token": token
                        })
                    
                    who = name or email_query
                    msg = f"Found {len(booking_records)} booking(s) for {who}:\n\n"
                    for i, booking in enumerate(booking_details, 1):
                        msg += f"{i}. **Booking ID:** {booking['id']}\n"
                        msg += f"   - **Price:** ${booking['price']:.2f}\n"
                        msg += f"   - **Status:** {booking['status']}\n"
                        msg += f"   - **Persons:** {booking['persons']}\n"
                        msg += f"   - **Token:** {booking['token']}\n\n"
                    
                    msg += f"**Total:** ${total_price:.2f} for {persons_total} persons"
                    
                    return Response({
                        "source": "site",
                        "message": msg,
                        "results": booking_records[:5],
                    })
                
                # Check if this is a payments query
                elif ("payment" in lowered or "payments" in lowered):
                    # Filter only payment records
                    payment_records = [h for h in matched if "payment" in h.get("title", "").lower()]
                    if not payment_records:
                        # If no direct payment records, look for them in all hits
                        for h in hits:
                            if "payment" in h.get("title", "").lower():
                                # Check if this payment is related to our customer
                                snip = h.get("snippet") or ""
                                for bid in booking_ids:
                                    if f"customerbookingid: {bid}" in snip.lower():
                                        payment_records.append(h)
                                        break
                    
                    if not payment_records:
                        who = name or email_query
                        return Response({
                            "source": "site",
                            "message": f"No payments found for {who}.",
                            "results": [],
                        })
                    
                    # Create detailed payment list
                    payment_details = []
                    total_amount = 0.0
                    
                    for h in payment_records:
                        snip = h.get("snippet") or ""
                        payment_id = ""
                        amount = 0.0
                        status = ""
                        gateway = ""
                        gateway_title = ""
                        date_time = ""
                        booking_id = ""
                        
                        # Extract payment details
                        for m in re.findall(r"id:\s*(\d+)", snip):
                            payment_id = m
                            break
                        for ma in re.findall(r"amount:\s*([0-9]+(?:\.[0-9]{1,2})?)", snip, flags=re.I):
                            try:
                                amount = float(ma)
                                total_amount += amount
                            except Exception:
                                pass
                        for ms in re.findall(r"status:\s*([^\n]+)", snip, flags=re.I):
                            status = ms.strip()
                        for mg in re.findall(r"gateway:\s*([^\n]+)", snip, flags=re.I):
                            gateway = mg.strip()
                        for mgt in re.findall(r"gatewaytitle:\s*([^\n]+)", snip, flags=re.I):
                            gateway_title = mgt.strip()
                        for mdt in re.findall(r"datetime:\s*([^\n]+)", snip, flags=re.I):
                            date_time = mdt.strip()
                        for mcb in re.findall(r"customerbookingid:\s*(\d+)", snip, flags=re.I):
                            booking_id = mcb.strip()
                        
                        payment_details.append({
                            "id": payment_id,
                            "amount": amount,
                            "status": status,
                            "gateway": gateway,
                            "gateway_title": gateway_title,
                            "date_time": date_time,
                            "booking_id": booking_id
                        })
                    
                    who = name or email_query
                    msg = f"Found {len(payment_records)} payment(s) for {who}:\n\n"
                    for i, payment in enumerate(payment_details, 1):
                        msg += f"{i}. **Payment ID:** {payment['id']}\n"
                        msg += f"   - **Amount:** ${payment['amount']:.2f}\n"
                        msg += f"   - **Status:** {payment['status']}\n"
                        msg += f"   - **Gateway:** {payment['gateway_title'] or payment['gateway']}\n"
                        msg += f"   - **Date:** {payment['date_time']}\n"
                        msg += f"   - **Booking ID:** {payment['booking_id']}\n\n"
                    
                    msg += f"**Total Amount:** ${total_amount:.2f}"
                    
                    return Response({
                        "source": "site",
                        "message": msg,
                        "results": payment_records[:5],
                    })
                
                # General user lookup
                else:
                    # Extract key fields
                    emails = []
                    fullnames = []
                    for h in matched:
                        snip = h.get("snippet") or ""
                        for me in re.findall(r"email:\s*([^\n]+)", snip, flags=re.I):
                            emails.append(me.strip())
                        for me2 in re.findall(r"emaillower:\s*([^\n]+)", snip, flags=re.I):
                            emails.append(me2.strip())
                        for mn in re.findall(r"fullname:\s*([^\n]+)", snip, flags=re.I):
                            fullnames.append(mn.strip())
                        for mn2 in re.findall(r"fullnamelower:\s*([^\n]+)", snip, flags=re.I):
                            fullnames.append(mn2.strip())
                    emails = list(dict.fromkeys(emails))
                    fullnames = list(dict.fromkeys(fullnames))
                    who = name or email_query
                    msg_parts = [f"Found {len(matched)} user record(s) for {who}."]
                    if fullnames:
                        msg_parts.append(f"Name: {fullnames[0]}")
                    if emails:
                        msg_parts.append(f"Email: {emails[0]}")
                    return Response({
                        "source": "site",
                        "message": " ".join(msg_parts),
                        "results": matched[:3],
                    })

        # Ask OpenAI to summarize the retrieved snippets into a concise answer
        # Use dynamic settings from WordPress database
        context = "\n\n".join([f"Title: {h.get('title') or ''}\nSnippet:\n{h.get('snippet') or ''}" for h in hits[:5]])
        prompt = (
            "Using ONLY the context below from our website, answer the user's question in a concise, helpful way. "
            "If details like date, time, price, or location are present, extract and include them exactly as stated. "
            "Do not say that no content was found if context is provided. If truly insufficient, say so briefly.\n\n"
            f"Context:\n{context}\n\nQuestion: {query}"
        )
        answer = openai_fallback(prompt, tone=tone, emojis=emojis, preface_no_site=False)
        return Response({
            "source": "site+openai",
            "message": answer,
            "results": hits[:3],
        })

    # 2) OpenAI fallback
    # Use dynamic settings from WordPress database (already loaded above)
    answer = openai_fallback(query, tone=tone, emojis=emojis, preface_no_site=True)
    return Response({
        "source": "openai",
        "answer": answer,
    })

# Create your views here.
