"""
WordPress utilities for reading settings from the WordPress database
"""
import logging
from django.db import connection
from django.conf import settings

logger = logging.getLogger(__name__)

def get_wp_option(option_name, default_value=None):
    """
    Get a WordPress option value from the wp_options table
    
    Args:
        option_name (str): The option name to retrieve
        default_value: Default value if option not found
        
    Returns:
        The option value or default_value if not found
    """
    try:
        with connection.cursor() as cursor:
            # Query the WordPress options table
            cursor.execute(
                "SELECT option_value FROM joxd_options WHERE option_name = %s",
                [option_name]
            )
            result = cursor.fetchone()
            
            if result:
                value = result[0]
                # Convert string '0'/'1' to boolean for emoji settings
                if option_name in ['acp_enable_emoji', 'acp_emoji_toggle']:
                    return value == '1' or value == 1
                return value
            else:
                logger.warning(f"WordPress option '{option_name}' not found, using default: {default_value}")
                return default_value
                
    except Exception as e:
        logger.error(f"Error reading WordPress option '{option_name}': {str(e)}")
        return default_value

def get_chatbot_settings():
    """
    Get all chatbot-related settings from WordPress
    
    Returns:
        dict: Dictionary containing chatbot settings
    """
    settings_dict = {
        'enable_emoji': get_wp_option('acp_enable_emoji', False),
        'enable_openai': get_wp_option('acp_enable_openai', True),
        'enable_woocommerce': get_wp_option('acp_enable_woocommerce', False),
        'enable_amelia': get_wp_option('acp_enable_amelia', False),
        'enable_faq': get_wp_option('acp_enable_faq', True),
        'enable_coupons': get_wp_option('acp_enable_coupons', False),
        'enable_rebooking': get_wp_option('acp_enable_rebooking', False),
        'enable_refunds': get_wp_option('acp_enable_refunds', False),
        'enable_cancellations': get_wp_option('acp_enable_cancellations', False),
        'enable_email': get_wp_option('acp_enable_email', False),
        'tone_of_voice': get_wp_option('acp_chatbot_tone', 'informative'),
    }
    
    return settings_dict

def get_chatbot_tone():
    """
    Get the chatbot tone setting from WordPress
    
    Returns:
        str: The tone setting or default
    """
    return get_wp_option('acp_chatbot_tone', 'informative')

def should_use_emojis():
    """
    Check if emojis should be used based on WordPress settings
    
    Returns:
        bool: True if emojis should be used, False otherwise
    """
    return get_wp_option('acp_enable_emoji', False)
    
