/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.core.internal.retry;

import java.time.Duration;
import java.util.OptionalDouble;
import java.util.function.Predicate;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.internal.retry.RateLimitingTokenBucket;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.core.retry.RetryPolicy;
import software.amazon.awssdk.core.retry.RetryPolicyContext;
import software.amazon.awssdk.core.retry.RetryUtils;
import software.amazon.awssdk.retries.api.AcquireInitialTokenRequest;
import software.amazon.awssdk.retries.api.AcquireInitialTokenResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.retries.api.RecordSuccessRequest;
import software.amazon.awssdk.retries.api.RecordSuccessResponse;
import software.amazon.awssdk.retries.api.RefreshRetryTokenRequest;
import software.amazon.awssdk.retries.api.RefreshRetryTokenResponse;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.retries.api.RetryToken;
import software.amazon.awssdk.retries.api.TokenAcquisitionFailedException;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public final class RetryPolicyAdapter
implements RetryStrategy {
    private final RetryPolicy retryPolicy;
    private final RetryPolicyContext retryPolicyContext;
    private final RateLimitingTokenBucket rateLimitingTokenBucket;

    private RetryPolicyAdapter(Builder builder) {
        this.retryPolicy = (RetryPolicy)Validate.paramNotNull((Object)builder.retryPolicy, (String)"retryPolicy");
        this.retryPolicyContext = builder.retryPolicyContext;
        this.rateLimitingTokenBucket = builder.rateLimitingTokenBucket;
    }

    public AcquireInitialTokenResponse acquireInitialToken(AcquireInitialTokenRequest request) {
        this.validateState();
        RetryPolicyAdapterToken token = new RetryPolicyAdapterToken(request.scope());
        return AcquireInitialTokenResponse.create((RetryToken)token, (Duration)this.rateLimitingTokenAcquire());
    }

    public RefreshRetryTokenResponse refreshRetryToken(RefreshRetryTokenRequest request) {
        this.validateState();
        RetryPolicyAdapterToken token = this.getToken(request.token());
        boolean willRetry = this.retryPolicy.aggregateRetryCondition().shouldRetry(this.retryPolicyContext);
        if (!willRetry) {
            this.retryPolicy.aggregateRetryCondition().requestWillNotBeRetried(this.retryPolicyContext);
            throw new TokenAcquisitionFailedException("Retry policy disallowed retry");
        }
        Duration backoffDelay = this.backoffDelay();
        return RefreshRetryTokenResponse.create((RetryToken)token, (Duration)backoffDelay);
    }

    public RecordSuccessResponse recordSuccess(RecordSuccessRequest request) {
        this.validateState();
        RetryPolicyAdapterToken token = this.getToken(request.token());
        this.retryPolicy.aggregateRetryCondition().requestSucceeded(this.retryPolicyContext);
        return RecordSuccessResponse.create((RetryToken)token);
    }

    public int maxAttempts() {
        return this.retryPolicy.numRetries() + 1;
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public boolean isInitialized() {
        return this.retryPolicyContext != null;
    }

    void validateState() {
        if (this.retryPolicyContext == null) {
            throw new IllegalStateException("This RetryPolicyAdapter instance has not been initialized.");
        }
    }

    RetryPolicyAdapterToken getToken(RetryToken token) {
        return (RetryPolicyAdapterToken)Validate.isInstanceOf(RetryPolicyAdapterToken.class, (Object)token, (String)"Object of class %s was not created by this retry strategy", (Object[])new Object[]{token.getClass().getName()});
    }

    boolean isFastFailRateLimiting() {
        return Boolean.TRUE.equals(this.retryPolicy.isFastFailRateLimiting());
    }

    Duration rateLimitingTokenAcquire() {
        if (!this.isRateLimitingEnabled()) {
            return Duration.ZERO;
        }
        OptionalDouble tokenAcquireTimeSeconds = this.rateLimitingTokenBucket.acquireNonBlocking(1.0, this.isFastFailRateLimiting());
        if (!tokenAcquireTimeSeconds.isPresent()) {
            String message = "Unable to acquire a send token immediately without waiting. This indicates that ADAPTIVE retry mode is enabled, fast fail rate limiting is enabled, and that rate limiting is engaged because of prior throttled requests. The request will not be executed.";
            throw new TokenAcquisitionFailedException(message, (Throwable)SdkClientException.create(message));
        }
        long tokenAcquireTimeMillis = (long)(tokenAcquireTimeSeconds.getAsDouble() * 1000.0);
        return Duration.ofMillis(tokenAcquireTimeMillis);
    }

    boolean isRateLimitingEnabled() {
        return this.retryPolicy.retryMode() == RetryMode.ADAPTIVE;
    }

    boolean isLastExceptionThrottlingException() {
        SdkException lastException = this.retryPolicyContext.exception();
        if (lastException == null) {
            return false;
        }
        return RetryUtils.isThrottlingException(lastException);
    }

    Duration backoffDelay() {
        Duration backoffDelay = RetryUtils.isThrottlingException(this.retryPolicyContext.exception()) ? this.retryPolicy.throttlingBackoffStrategy().computeDelayBeforeNextRetry(this.retryPolicyContext) : this.retryPolicy.backoffStrategy().computeDelayBeforeNextRetry(this.retryPolicyContext);
        Duration rateLimitingDelay = this.rateLimitingTokenAcquire();
        return backoffDelay.plus(rateLimitingDelay);
    }

    public static Builder builder() {
        return new Builder();
    }

    static class RetryPolicyAdapterToken
    implements RetryToken {
        private final String scope;

        RetryPolicyAdapterToken(String scope) {
            this.scope = (String)Validate.paramNotNull((Object)scope, (String)"scope");
        }
    }

    public static class Builder
    implements RetryStrategy.Builder<Builder, RetryPolicyAdapter> {
        private RetryPolicy retryPolicy;
        private RetryPolicyContext retryPolicyContext;
        private RateLimitingTokenBucket rateLimitingTokenBucket;

        private Builder() {
        }

        private Builder(RetryPolicyAdapter adapter) {
            this.retryPolicy = adapter.retryPolicy;
            this.retryPolicyContext = adapter.retryPolicyContext;
            this.rateLimitingTokenBucket = adapter.rateLimitingTokenBucket;
        }

        public Builder retryOnException(Predicate<Throwable> shouldRetry) {
            throw new UnsupportedOperationException("RetryPolicyAdapter does not support calling retryOnException");
        }

        public Builder maxAttempts(int maxAttempts) {
            throw new UnsupportedOperationException("RetryPolicyAdapter does not support calling maxAttempts");
        }

        public Builder backoffStrategy(BackoffStrategy backoffStrategy) {
            throw new UnsupportedOperationException("RetryPolicyAdapter does not support calling backoffStrategy");
        }

        public Builder throttlingBackoffStrategy(BackoffStrategy backoffStrategy) {
            throw new UnsupportedOperationException("RetryPolicyAdapter does not support calling throttlingBackoffStrategy");
        }

        public Builder treatAsThrottling(Predicate<Throwable> treatAsThrottling) {
            throw new UnsupportedOperationException("RetryPolicyAdapter does not support calling treatAsThrottling");
        }

        public Builder retryPolicy(RetryPolicy retryPolicy) {
            this.retryPolicy = retryPolicy;
            return this;
        }

        public Builder retryPolicyContext(RetryPolicyContext retryPolicyContext) {
            this.retryPolicyContext = retryPolicyContext;
            return this;
        }

        public Builder initialize(RetryPolicyContext retryPolicyContext) {
            this.retryPolicyContext = retryPolicyContext;
            this.rateLimitingTokenBucket = new RateLimitingTokenBucket();
            return this;
        }

        public RetryPolicyAdapter build() {
            return new RetryPolicyAdapter(this);
        }
    }
}

