# Django Vapi Integration - Last Conversation Endpoint

This Django project provides an API endpoint for Vapi.ai to fetch the last conversation from your PostgreSQL database.

## Setup

### 1. Install Dependencies

```bash
source venv/bin/activate
pip install -r requirements.txt
```

### 2. Configure PostgreSQL Database

Set the following environment variables or update `config/settings.py`:

```bash
export DB_NAME=your_database_name
export DB_USER=your_database_user
export DB_PASSWORD=your_database_password
export DB_HOST=localhost
export DB_PORT=5432
```

Or update the `DATABASES` setting in `config/settings.py` directly.

### 3. Run Migrations

```bash
python manage.py makemigrations
python manage.py migrate
```

### 4. Create Superuser (Optional)

```bash
python manage.py createsuperuser
```

### 5. Run Development Server

```bash
python manage.py runserver
```

## API Endpoint

### Get Last Conversation

**URL:** `POST /api/calls/get_last_conversation/`

**Request Body:**
```json
{
  "phone_number": "+49123456789"
}
```

**Response:**
```json
{
  "context": "Summary of last conversation: ...\nTopics discussed: ...",
  "has_previous_conversation": true,
  "last_conversation_date": "2024-01-15T10:30:00Z",
  "summary": "...",
  "topics": ["topic1", "topic2"],
  "emotions": ["happy", "calm"],
  "key_points": ["point1", "point2"]
}
```

**No Previous Conversation Response:**
```json
{
  "context": "This is the first conversation with this person.",
  "has_previous_conversation": false,
  "last_conversation_date": null,
  "summary": "",
  "topics": [],
  "emotions": []
}
```

## Vapi Configuration

### Option 1: Using Server URL (Function Calling)

1. Go to Vapi Dashboard → Your Assistant
2. Navigate to "Server URL" or "Function Calling"
3. Add a new function:
   - Function Name: `get_last_conversation`
   - Description: "Get previous conversation context for personalization"
   - Method: `POST`
   - URL: `https://your-domain.com/api/calls/get_last_conversation/`
   - Headers:
     ```json
     {
       "Content-Type": "application/json"
     }
     ```
   - Request Body Schema:
     ```json
     {
       "type": "object",
       "properties": {
         "phone_number": {
           "type": "string",
           "description": "The phone number being called"
         }
       },
       "required": ["phone_number"]
     }
     ```

4. In your System Message, add:
```
You are Clara, a 72-year-old friend from Monheim am Rhein...

Previous conversation context:
{{get_last_conversation(customer.phoneNumber)}}
```

### Option 2: Using Vapi's Function Calling Feature

1. In Assistant Settings → Functions:
   - Click "Add Function"
   - Name: `get_last_conversation`
   - Description: "Retrieve last conversation context"
   - Server URL: `https://your-domain.com/api/calls/get_last_conversation/`
   - Parameters:
     ```json
     {
       "phone_number": {
         "type": "string",
         "description": "Phone number of the person being called"
       }
     }
     ```

2. In System Message:
```
You are Clara...

{{#if get_last_conversation}}
Previous conversation:
{{get_last_conversation(customer.phoneNumber).context}}
{{/if}}
```

## Testing

Test the endpoint locally:

```bash
curl -X POST http://localhost:8000/api/calls/get_last_conversation/ \
  -H "Content-Type: application/json" \
  -d '{"phone_number": "+49123456789"}'
```

## Database Models

### Call Model
- `phone_number`: Phone number of the caller
- `call_id`: Unique call identifier
- `started_at`: When the call started
- `ended_at`: When the call ended
- `duration_seconds`: Call duration
- `status`: Call status

### Conversation Model
- `call`: One-to-one relationship with Call
- `transcript`: Full conversation transcript
- `summary`: Summary of the conversation
- `topics`: JSON array of topics discussed
- `emotions`: JSON array of emotions detected
- `key_points`: JSON array of key points

## Security Notes

- The endpoint currently allows any requests (`AllowAny` permission)
- For production, consider adding:
  - API key authentication
  - Rate limiting
  - IP whitelisting for Vapi's IPs
  - HTTPS only

## Deployment

1. Set `DEBUG = False` in `config/settings.py`
2. Update `ALLOWED_HOSTS` with your domain
3. Configure proper database credentials
4. Set up static files serving
5. Use a production WSGI server (e.g., Gunicorn)
6. Set up reverse proxy (e.g., Nginx)

