# Senior Engagement App

A compassionate, AI-driven telephony application designed to enrich the lives of seniors through meaningful, memory-preserving conversations. This Django application integrates with **Vapi.ai** for voice AI orchestration and **Twilio** for phone connectivity, backed by a **PostgreSQL** database that intelligently retains conversation history to personalize future calls.

## 🌟 Features

- **AI-Powered Conversations**: Vapi.ai voice agent conducts warm, engaging calls with seniors
- **German Language Support**: Full German language support with German voice (de-DE-KatjaNeural) and German prompts
- **Memory System**: Every conversation is analyzed and stored for future personalization
- **Twilio Integration**: Reliable phone connectivity for outbound calls
- **Conversation Analytics**: Track sentiment, engagement levels, and conversation topics
- **Modern Web Interface**: Beautiful, responsive UI with light/dark theme support
- **Call History with Pagination**: View and navigate through call history efficiently
- **Auto-refresh**: Real-time updates when calls complete
- **REST API**: Full API for integration with other systems
- **Admin Interface**: Django admin for managing data

## 🏗️ Architecture

```
User Interface → Django REST API → Celery Tasks → Twilio/Vapi.ai → PostgreSQL
```

### Data Flow
1. Caregiver adds senior profile via web interface
2. Caregiver initiates call through API
3. Django creates call record and triggers Celery task
4. Celery task calls Twilio to place outbound call
5. Twilio connects to Vapi.ai voice agent
6. Vapi.ai conducts conversation using personalized prompts
7. Post-call, Vapi.ai sends transcript/summary via webhook
8. Django processes webhook and stores conversation data
9. Future calls use stored memories for personalization

## 🚀 Quick Start

### Prerequisites

- Python 3.8+
- PostgreSQL 12+
- Redis (for Celery)
- Twilio account
- Vapi.ai account

### Starting Celery Services

The application includes a convenient script to start Celery worker and beat:

```bash
bash start_celery.sh
```

This will start:
- Celery Worker: Processes background tasks (call initiation, status updates)
- Celery Beat: Runs periodic tasks (checking call statuses every 10 seconds)

Monitor logs:
- Worker logs: `tail -f /tmp/celery_worker.log`
- Beat logs: `tail -f /tmp/celery_beat.log`

### Installation

1. **Clone the repository**
   ```bash
   git clone <repository-url>
   cd senior-engagement-app
   ```

2. **Create virtual environment**
   ```bash
   python -m venv venv
   source venv/bin/activate  # On Windows: venv\Scripts\activate
   ```

3. **Install dependencies**
   ```bash
   pip install -r requirements.txt
   ```

4. **Set up environment variables**
   ```bash
   cp env.example .env
   # Edit .env with your configuration
   ```

5. **Set up PostgreSQL database**
   ```sql
   CREATE DATABASE senior_engagement;
   CREATE USER postgres WITH PASSWORD 'password';
   GRANT ALL PRIVILEGES ON DATABASE senior_engagement TO postgres;
   ```

6. **Run migrations**
   ```bash
   python manage.py makemigrations
   python manage.py migrate
   ```

7. **Create superuser**
   ```bash
   python manage.py createsuperuser
   ```

8. **Start Redis server**
   ```bash
   redis-server
   ```

9. **Start Celery services** (using the provided script)
   ```bash
   bash start_celery.sh
   ```
   
   Or manually:
   ```bash
   # Terminal 1: Celery Worker
   celery -A senior_engagement_app worker --loglevel=info
   
   # Terminal 2: Celery Beat (optional, for periodic tasks)
   celery -A senior_engagement_app beat --loglevel=info
   ```

10. **Start Django development server**
    ```bash
    python manage.py runserver
    ```

11. **Access the application**
    - Web Interface: http://localhost:8000
    - Admin Interface: http://localhost:8000/admin
    - API Documentation: http://localhost:8000/api/

## ⚙️ Configuration

### Environment Variables

Create a `.env` file with the following variables:

```env
# Django Configuration
SECRET_KEY=your-secret-key-here
DEBUG=True
ALLOWED_HOSTS=localhost,127.0.0.1

# Database Configuration
DB_NAME=senior_engagement
DB_USER=postgres
DB_PASSWORD=password
DB_HOST=localhost
DB_PORT=5432

# Redis Configuration
REDIS_URL=redis://localhost:6379/0

# Twilio Configuration
TWILIO_ACCOUNT_SID=your-twilio-account-sid
TWILIO_AUTH_TOKEN=your-twilio-auth-token
TWILIO_PHONE_NUMBER=your-twilio-phone-number

# Vapi.ai Configuration
VAPI_API_KEY=your-vapi-api-key
VAPI_ASSISTANT_ID=your-vapi-assistant-id

# Base URL for webhooks
BASE_URL=http://localhost:8000
```

### Twilio Setup

1. Sign up for a [Twilio account](https://www.twilio.com/)
2. Get your Account SID and Auth Token from the console
3. Purchase a phone number for outbound calls
4. Configure webhook URLs in your Twilio console

### Vapi.ai Setup

1. Sign up for a [Vapi.ai account](https://vapi.ai/)
2. Create a voice assistant
3. Get your API key and Assistant ID
4. Configure webhook URLs in your Vapi.ai dashboard

## 📱 API Endpoints

### Seniors
- `GET /api/seniors/` - List all seniors
- `POST /api/seniors/` - Create a new senior
- `GET /api/seniors/{id}/` - Get senior details
- `PUT /api/seniors/{id}/` - Update senior
- `DELETE /api/seniors/{id}/` - Delete senior
- `POST /api/seniors/{id}/initiate_call/` - Initiate call to senior
- `GET /api/seniors/{id}/conversation_history/` - Get conversation history
- `GET /api/seniors/{id}/memories/` - Get senior's memories
- `GET /api/seniors/{id}/insights/` - Get conversation insights

### Calls
- `GET /api/calls/` - List all calls
- `GET /api/calls/{id}/` - Get call details
- `GET /api/calls/{id}/logs/` - Get call logs
- `POST /api/calls/twilio_webhook/` - Twilio webhook endpoint
- `POST /api/calls/vapi_webhook/` - Vapi.ai webhook endpoint
- `GET /api/calls/stats/` - Get call statistics

### Conversations
- `GET /api/conversations/` - List all conversations
- `GET /api/conversations/{id}/` - Get conversation details
- `GET /api/conversations/{id}/memories/` - Get conversation memories
- `GET /api/conversations/analytics/` - Get conversation analytics

### Memories
- `GET /api/memories/` - List all memories
- `GET /api/memories/by_type/` - Filter memories by type
- `GET /api/memories/by_senior/` - Filter memories by senior

### Insights
- `GET /api/insights/` - List all insights
- `GET /api/insights/by_type/` - Filter insights by type

## 🗄️ Database Schema

### Core Models

#### Senior
- Basic information (name, phone, age)
- Preferences (call times, interests, health notes)
- Relationship to caregiver
- Conversation summary for personalization

#### Call
- Twilio and Vapi.ai integration data
- Call timing and duration
- AI prompt used for the call
- Status tracking

#### Conversation
- Full transcript and AI-generated summary
- Topics discussed and emotions detected
- Sentiment analysis and engagement level
- Follow-up topics and questions

#### Memory
- Extracted memories from conversations
- Importance scoring and usage tracking
- Categorization by type (story, fact, preference, etc.)

#### ConversationInsight
- Derived insights from conversation analysis
- Confidence scoring
- Trend analysis and pattern recognition

## 🤖 AI Personalization

The system uses conversation history to personalize future calls:

1. **Memory Extraction**: Important stories and facts are extracted from each conversation
2. **Context Building**: Recent conversation summaries are included in AI prompts
3. **Follow-up Questions**: System generates relevant questions based on past topics
4. **Sentiment Tracking**: Monitors emotional trends over time
5. **Interest Evolution**: Tracks changes in topics of interest

### Language Support

The system currently supports **German** language:
- **German Voice**: Uses Azure TTS `de-DE-KatjaNeural` (German female voice)
- **German Prompts**: All AI prompts and instructions are in German
- **German Greetings**: Personalized greetings are generated in German
- **Dementia-Friendly Communication**: Guidelines adapted for German-speaking seniors

To change the language, modify the `generate_ai_prompt()` function in `calls/tasks.py` and update the voice configuration in `create_vapi_assistant_with_prompt()`.

### Example AI Prompt (German)
```
Sie sind ein freundlicher, geduldiger und einfühlsamer Gesprächspartner für Menschen 
mit früher oder fortgeschrittener Demenz. Sie müssen IMMER auf Deutsch sprechen. 
Sprechen Sie langsam, deutlich und verwenden Sie einfache, kurze Sätze...

Informationen zum Senioren:
- Name: Max Müller
- Alter: 78
- Interessen: Gartenarbeit, Kochen
- Gesundheitshinweise: Keine

Kontext aus vorherigen Gesprächen:
Zusammenfassung des letzten Gesprächs: Max erwähnte, dass seine Tomatenpflanzen 
gut wachsen und er vorhatte, sein berühmtes Lasagne-Rezept zu kochen.
```

## 🔧 Development

### Running Tests
```bash
python manage.py test
```

### Code Quality
```bash
# Install development dependencies
pip install black flake8 isort

# Format code
black .

# Check code style
flake8 .

# Sort imports
isort .
```

### Database Management
```bash
# Create migrations
python manage.py makemigrations

# Apply migrations
python manage.py migrate

# Reset database (development only)
python manage.py flush
```

## 🚀 Deployment

### Production Checklist

1. **Environment Variables**
   - Set `DEBUG=False`
   - Use strong `SECRET_KEY`
   - Configure production database
   - Set up proper `ALLOWED_HOSTS`

2. **Database**
   - Use PostgreSQL in production
   - Set up database backups
   - Configure connection pooling

3. **Static Files**
   ```bash
   python manage.py collectstatic
   ```

4. **Celery**
   - Use Redis or RabbitMQ as broker
   - Set up Celery workers
   - Configure monitoring

5. **Webhooks**
   - Use HTTPS URLs for webhooks
   - Set up proper authentication
   - Monitor webhook delivery

### Docker Deployment

```dockerfile
FROM python:3.10-slim

WORKDIR /app
COPY requirements.txt .
RUN pip install -r requirements.txt

COPY . .
RUN python manage.py collectstatic --noinput

EXPOSE 8000
CMD ["gunicorn", "senior_engagement_app.wsgi:application"]
```

## 📊 Monitoring

### Key Metrics to Monitor

- Call success rate
- Average call duration
- Conversation sentiment trends
- Memory extraction accuracy
- Webhook delivery success
- System performance metrics

### Logging

The application logs important events:
- Call initiation and completion
- Webhook processing
- Error conditions
- Performance metrics

## 🤝 Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests
5. Submit a pull request

## 📄 License

This project is licensed under the MIT License - see the LICENSE file for details.

## 🆘 Support

For support and questions:
- Create an issue in the repository
- Check the documentation
- Review the API endpoints

## 🎨 Recent Improvements

### UI/UX Enhancements
- **Modern Design**: Clean, elegant interface inspired by professional healthcare applications
- **Theme System**: Light/dark mode with persistent preferences
- **Enhanced Seniors Cards**: Avatar initials, soft badges, improved hover effects
- **Call History Cards**: Better visual hierarchy with avatars, status badges, and duration display
- **Pagination**: Efficient navigation through call history
- **Auto-refresh**: Real-time updates when calls complete
- **Responsive Design**: Works seamlessly on all device sizes

### Functionality Improvements
- **Call Duration Tracking**: Fixed duration calculation and display
- **Button State Management**: Proper loading states and automatic restoration
- **Error Handling**: Better error messages and user feedback
- **Performance**: Optimized API calls and data loading

### Language Support
- **German Language**: Full support for German conversations
  - German voice (de-DE-KatjaNeural)
  - German prompts and instructions
  - German greetings

See `RECENT_IMPROVEMENTS.md` for detailed changelog.

## 🙏 Acknowledgments

- **Twilio** for reliable telephony infrastructure
- **Vapi.ai** for AI voice agent capabilities
- **Django** for the robust web framework
- **PostgreSQL** for reliable data storage
- **Azure Cognitive Services** for German TTS voices

---

**Built with ❤️ for seniors and their families**
