from django.db import models
from django.contrib.auth.models import User
from seniors.models import Senior


class Call(models.Model):
    """Model representing a call made to a senior"""
    
    STATUS_CHOICES = [
        ('initiated', 'Initiated'),
        ('ringing', 'Ringing'),
        ('answered', 'Answered'),
        ('completed', 'Completed'),
        ('failed', 'Failed'),
        ('busy', 'Busy'),
        ('no_answer', 'No Answer'),
        ('cancelled', 'Cancelled'),
    ]
    
    # Call Information
    senior = models.ForeignKey(
        Senior,
        on_delete=models.CASCADE,
        related_name='calls',
        help_text="The senior who received the call"
    )
    
    # Twilio Information
    twilio_sid = models.CharField(
        max_length=100,
        unique=True,
        null=True,
        blank=True,
        help_text="Twilio Call SID"
    )
    twilio_status = models.CharField(
        max_length=20,
        choices=STATUS_CHOICES,
        default='initiated',
        help_text="Status of the call"
    )
    
    # Call Details
    duration = models.PositiveIntegerField(
        null=True,
        blank=True,
        help_text="Call duration in seconds"
    )
    call_start_time = models.DateTimeField(
        null=True,
        blank=True,
        help_text="When the call actually started"
    )
    call_end_time = models.DateTimeField(
        null=True,
        blank=True,
        help_text="When the call ended"
    )
    
    # Vapi Information
    vapi_call_id = models.CharField(
        max_length=100,
        null=True,
        blank=True,
        help_text="Vapi.ai call ID"
    )
    vapi_assistant_id = models.CharField(
        max_length=100,
        null=True,
        blank=True,
        help_text="Vapi.ai assistant ID created for this call"
    )
    vapi_status = models.CharField(
        max_length=50,
        null=True,
        blank=True,
        help_text="Vapi.ai call status"
    )
    
    # Call Configuration
    ai_prompt = models.TextField(
        blank=True,
        help_text="The AI prompt used for this call (includes memory context)"
    )
    call_purpose = models.CharField(
        max_length=200,
        default='general_checkin',
        help_text="Purpose of the call (e.g., 'general_checkin', 'birthday_call')"
    )
    
    # Metadata
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    initiated_by = models.ForeignKey(
        User,
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        help_text="User who initiated the call"
    )
    
    class Meta:
        ordering = ['-created_at']
    
    def __str__(self):
        return f"Call to {self.senior.name} - {self.twilio_status} ({self.created_at.strftime('%Y-%m-%d %H:%M')})"
    
    @property
    def status(self):
        """Return the current status of the call"""
        return self.twilio_status
    
    @property
    def is_successful(self):
        """Return True if the call was successful"""
        return self.twilio_status == 'completed'
    
    @property
    def formatted_duration(self):
        """Return formatted duration string"""
        if not self.duration:
            return "N/A"
        
        minutes = self.duration // 60
        seconds = self.duration % 60
        return f"{minutes}m {seconds}s"


class CallLog(models.Model):
    """Model for logging call events and status changes"""
    
    EVENT_TYPES = [
        ('initiated', 'Call Initiated'),
        ('twilio_created', 'Twilio Call Created'),
        ('vapi_started', 'Vapi Call Started'),
        ('answered', 'Call Answered'),
        ('completed', 'Call Completed'),
        ('failed', 'Call Failed'),
        ('status_changed', 'Status Changed'),
        ('webhook_received', 'Webhook Received'),
    ]
    
    call = models.ForeignKey(
        Call,
        on_delete=models.CASCADE,
        related_name='logs',
        help_text="The call this log entry belongs to"
    )
    
    event_type = models.CharField(
        max_length=20,
        choices=EVENT_TYPES,
        help_text="Type of event logged"
    )
    
    message = models.TextField(help_text="Log message")
    data = models.JSONField(
        default=dict,
        help_text="Additional data related to the event"
    )
    
    created_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-created_at']
    
    def __str__(self):
        return f"{self.call} - {self.event_type}: {self.message[:50]}"