from django.db import models
from django.contrib.auth.models import User
from django.core.validators import RegexValidator


class Senior(models.Model):
    """Model representing a senior person who will receive calls"""
    
    # Basic Information
    name = models.CharField(max_length=100, help_text="Full name of the senior")
    phone_number = models.CharField(
        max_length=15,
        validators=[RegexValidator(
            regex=r'^\+?1?\d{9,15}$',
            message="Phone number must be entered in the format: '+999999999'. Up to 15 digits allowed."
        )],
        help_text="Phone number in international format (e.g., +1234567890)"
    )
    
    # Personal Details
    age = models.PositiveIntegerField(null=True, blank=True, help_text="Age of the senior")
    timezone = models.CharField(max_length=50, default='UTC', help_text="Timezone for scheduling calls")
    
    # Preferences
    preferred_call_times = models.JSONField(
        default=list,
        help_text="List of preferred call times (e.g., ['morning', 'afternoon'])"
    )
    interests = models.JSONField(
        default=list,
        help_text="List of interests and topics the senior enjoys discussing"
    )
    health_notes = models.TextField(
        blank=True,
        help_text="Any health-related notes or considerations for conversations"
    )
    
    # Relationship Information
    caregiver = models.ForeignKey(
        User,
        on_delete=models.CASCADE,
        related_name='seniors',
        help_text="The caregiver or family member managing this senior's profile"
    )
    
    # Metadata
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    is_active = models.BooleanField(default=True, help_text="Whether this senior is currently active")
    
    class Meta:
        ordering = ['name']
    
    def __str__(self):
        return f"{self.name} ({self.phone_number})"
    