from rest_framework import serializers
from django.contrib.auth.models import User
from .models import Senior


class UserSerializer(serializers.ModelSerializer):
    """Serializer for User model"""
    
    class Meta:
        model = User
        fields = ['id', 'username', 'first_name', 'last_name', 'email']


class SeniorSerializer(serializers.ModelSerializer):
    """Serializer for Senior model"""
    
    caregiver = UserSerializer(read_only=True)
    total_calls = serializers.ReadOnlyField()
    last_call_date = serializers.ReadOnlyField()
    
    class Meta:
        model = Senior
        fields = [
            'id', 'name', 'phone_number', 'age', 'timezone',
            'preferred_call_times', 'interests', 'health_notes',
            'caregiver', 'is_active', 'created_at', 'updated_at',
            'total_calls', 'last_call_date'
        ]
        read_only_fields = ['id', 'created_at', 'updated_at']


class SeniorDetailSerializer(SeniorSerializer):
    """Detailed serializer for Senior model with conversation summary"""
    
    conversation_summary = serializers.ReadOnlyField()
    
    class Meta(SeniorSerializer.Meta):
        fields = SeniorSerializer.Meta.fields + ['conversation_summary']


class SeniorCreateSerializer(serializers.ModelSerializer):
    """Serializer for creating Senior instances"""
    
    class Meta:
        model = Senior
        fields = [
            'name', 'phone_number', 'age', 'timezone',
            'preferred_call_times', 'interests', 'health_notes'
        ]
    
    def validate_phone_number(self, value):
        """Validate phone number format"""
        if not value.startswith('+'):
            raise serializers.ValidationError(
                "Phone number must include country code (e.g., +1234567890)"
            )
        return value
    
    def validate_preferred_call_times(self, value):
        """Validate preferred call times"""
        valid_times = ['morning', 'afternoon', 'evening', 'anytime']
        for time in value:
            if time not in valid_times:
                raise serializers.ValidationError(
                    f"Invalid call time: {time}. Must be one of: {', '.join(valid_times)}"
                )
        return value
