# Senior Engagement App

A compassionate, AI-driven telephony application designed to enrich the lives of seniors through meaningful, memory-preserving conversations. This Django application integrates with **Vapi.ai** for voice AI orchestration and **Twilio** for phone connectivity, backed by a **PostgreSQL** database that intelligently retains conversation history to personalize future calls.

## 🌟 Features

- **AI-Powered Conversations**: Vapi.ai voice agent conducts warm, engaging calls with seniors
- **Customizable Call Prompts**: Manage reusable prompt templates through the admin panel for different call scenarios
- **Memory System**: Every conversation is analyzed and stored for future personalization
- **Twilio Integration**: Reliable phone connectivity for outbound calls
- **Conversation Analytics**: Track sentiment, engagement levels, and conversation topics
- **Web Interface**: Simple interface for caregivers to manage seniors and initiate calls
- **REST API**: Full API for integration with other systems
- **Admin Interface**: Beautiful Django admin interface powered by Jazzmin
- **Celery Task Queue**: Asynchronous call processing and status updates
- **Call Logging**: Comprehensive logging of all call events and status changes

## 🏗️ Architecture

```
User Interface → Django REST API → Celery Tasks → Twilio/Vapi.ai → PostgreSQL
```

### Data Flow
1. Caregiver adds senior profile via web interface or API
2. Caregiver initiates call through API (optionally with a specific prompt)
3. Django creates call record and triggers Celery task
4. Celery task calls Twilio to place outbound call
5. Twilio connects to Vapi.ai voice agent
6. Vapi.ai conducts conversation using personalized prompts (with memory context)
7. Post-call, Vapi.ai sends transcript/summary via webhook
8. Django processes webhook and stores conversation data
9. Future calls use stored memories for personalization

## 🚀 Quick Start

### Prerequisites

- Python 3.8+
- PostgreSQL 12+
- Redis (for Celery)
- Twilio account
- Vapi.ai account

### Installation

1. **Clone the repository**
   ```bash
   git clone <repository-url>
   cd senior-engagement-app
   ```

2. **Create virtual environment**
   ```bash
   python -m venv venv
   source venv/bin/activate  # On Windows: venv\Scripts\activate
   ```

3. **Install dependencies**
   ```bash
   pip install -r requirements.txt
   ```

4. **Set up environment variables**
   
   Create a `.env` file in the project root:
   ```bash
   cp example_env .env
   # Edit .env with your configuration
   ```
   
   Required environment variables:
   ```env
   # Django Configuration
   SECRET_KEY=your-secret-key-here
   DEBUG=True
   ALLOWED_HOSTS=localhost,127.0.0.1
   
   # Database Configuration
   DB_NAME=senior_engagement
   DB_USER=postgres
   DB_PASSWORD=password
   DB_HOST=localhost
   DB_PORT=5432
   
   # Redis Configuration (for Celery)
   REDIS_URL=redis://localhost:6379/0
   
   # Twilio Configuration
   TWILIO_ACCOUNT_SID=your-twilio-account-sid
   TWILIO_AUTH_TOKEN=your-twilio-auth-token
   TWILIO_PHONE_NUMBER=your-twilio-phone-number
   
   # Vapi.ai Configuration
   VAPI_API_KEY=your-vapi-api-key
   VAPI_ASSISTANT_ID=your-vapi-assistant-id
   VAPI_PHONE_NUMBER_ID=your-vapi-phone-number-id
   
   # Base URL for webhooks
   BASE_URL=http://localhost:8000
   ```

5. **Set up PostgreSQL database**
   ```sql
   CREATE DATABASE senior_engagement;
   CREATE USER postgres WITH PASSWORD 'password';
   GRANT ALL PRIVILEGES ON DATABASE senior_engagement TO postgres;
   ```

6. **Run migrations**
   ```bash
   python manage.py makemigrations
   python manage.py migrate
   ```

7. **Create superuser**
   ```bash
   python manage.py createsuperuser
   ```

8. **Start Redis server**
   ```bash
   redis-server
   ```

9. **Start Celery worker**
   ```bash
   celery -A senior_engagement_app worker --loglevel=info
   ```
   
   Or use the provided script:
   ```bash
   ./start_celery.sh
   ```

10. **Start Celery Beat (for periodic tasks)**
    ```bash
    celery -A senior_engagement_app beat --loglevel=info
    ```

11. **Start Django development server**
    ```bash
    python manage.py runserver
    ```

12. **Access the application**
    - Web Interface: http://localhost:8000
    - Admin Interface: http://localhost:8000/admin
    - API Root: http://localhost:8000/api/

## ⚙️ Configuration

### Environment Variables

The application uses `python-decouple` to manage environment variables. All configuration is loaded from the `.env` file. See the `example_env` file for a template.

### Twilio Setup

1. Sign up for a [Twilio account](https://www.twilio.com/)
2. Get your Account SID and Auth Token from the console
3. Purchase a phone number for outbound calls
4. Configure webhook URLs in your Twilio console:
   - Status Callback: `http://your-domain/api/calls/twilio_webhook/`

### Vapi.ai Setup

1. Sign up for a [Vapi.ai account](https://vapi.ai/)
2. Create a voice assistant
3. Get your API key and Assistant ID
4. Configure webhook URLs in your Vapi.ai dashboard:
   - Webhook URL: `http://your-domain/api/calls/vapi_webhook/`

## 📱 API Endpoints

The API uses Django REST Framework with ViewSets, providing standard REST endpoints.

### Seniors

- `GET /api/seniors/` - List all active seniors
- `POST /api/seniors/` - Create a new senior
- `GET /api/seniors/{id}/` - Get senior details
- `PUT /api/seniors/{id}/` - Update senior
- `PATCH /api/seniors/{id}/` - Partially update senior
- `DELETE /api/seniors/{id}/` - Delete senior
- `POST /api/seniors/{id}/initiate_call/` - Initiate call to senior
  ```json
  {
    "purpose": "general_checkin",
    "prompt_id": 1  // Optional: ID of prompt to use
  }
  ```
- `GET /api/seniors/{id}/conversation_history/` - Get conversation history
- `GET /api/seniors/{id}/memories/` - Get senior's memories
- `GET /api/seniors/{id}/insights/` - Get conversation insights

### Calls

- `GET /api/calls/` - List all calls
- `GET /api/calls/{id}/` - Get call details
- `GET /api/calls/{id}/logs/` - Get call logs
- `POST /api/calls/twilio_webhook/` - Twilio webhook endpoint
- `POST /api/calls/vapi_webhook/` - Vapi.ai webhook endpoint

### Call Prompts

- `GET /api/prompts/` - List all active prompts
- `GET /api/prompts/{id}/` - Get specific prompt details

### Conversations

- `GET /api/conversations/` - List all conversations
- `GET /api/conversations/{id}/` - Get conversation details

### Memories

- `GET /api/memories/` - List all memories
- `GET /api/memories/{id}/` - Get memory details

### Insights

- `GET /api/insights/` - List all insights
- `GET /api/insights/{id}/` - Get insight details

## 🗄️ Database Schema

### Core Models

#### Senior
- Basic information (name, phone, age, timezone)
- Preferences (call times, interests, health notes)
- Relationship to caregiver
- Active status

#### CallPrompt
- Reusable prompt templates for different call scenarios
- Supports placeholders: `{senior_name}`, `{senior_age}`, `{interests}`, `{health_notes}`, `{call_purpose}`, `{conversation_context}`
- Active/inactive status
- Default prompt designation

#### Call
- Twilio and Vapi.ai integration data
- Call timing and duration
- AI prompt used for the call
- Status tracking (initiated, ringing, answered, completed, failed, etc.)
- Relationship to CallPrompt

#### CallLog
- Event logging for all call activities
- Tracks status changes, webhook events, and errors

#### Conversation
- Full transcript and AI-generated summary
- Topics discussed and emotions detected
- Sentiment analysis and engagement level
- Follow-up topics and questions
- One-to-one relationship with Call

#### Memory
- Extracted memories from conversations
- Importance scoring and usage tracking
- Categorization by type (story, fact, preference, etc.)
- Relationship to Senior and Conversation

#### ConversationInsight
- Derived insights from conversation analysis
- Confidence scoring
- Trend analysis and pattern recognition

## 🤖 AI Personalization

The system uses conversation history to personalize future calls:

1. **Memory Extraction**: Important stories and facts are extracted from each conversation
2. **Context Building**: Recent conversation summaries are included in AI prompts
3. **Prompt Templates**: Reusable prompts can be customized for different scenarios (check-ins, birthdays, etc.)
4. **Follow-up Questions**: System generates relevant questions based on past topics
5. **Sentiment Tracking**: Monitors emotional trends over time
6. **Interest Evolution**: Tracks changes in topics of interest

### Using Call Prompts

1. **Create Prompts in Admin**: Navigate to Admin → Calls → Call Prompts
2. **Use Placeholders**: Include placeholders like `{senior_name}`, `{conversation_context}`, etc.
3. **Set Default**: Mark one prompt as default to use when no prompt_id is specified
4. **Initiate Calls**: Use the `prompt_id` parameter when initiating calls via API

### Example AI Prompt with Context

```
You are calling John, a 78-year-old who enjoys gardening and cooking.
Last time, he mentioned his tomato plants were doing well and he was
planning to make his famous lasagna. Ask him about his garden and
if he's had a chance to make that lasagna yet.

[Previous conversation context is automatically included]
```

## 🔧 Development

### Running Tests
```bash
python manage.py test
```

### Code Quality
```bash
# Install development dependencies
pip install black flake8 isort

# Format code
black .

# Check code style
flake8 .

# Sort imports
isort .
```

### Database Management
```bash
# Create migrations
python manage.py makemigrations

# Apply migrations
python manage.py migrate

# Reset database (development only)
python manage.py flush
```

### Management Commands

The application includes several management commands:

- `clear_active_calls` - Clear stuck active calls
- `clear_stuck_calls` - Clear calls stuck in various states
- `fix_stuck_calls` - Fix calls that are stuck
- `generate_memories_insights` - Generate memories and insights from conversations
- `process_conversations` - Process conversations to extract memories
- `update_call_statuses` - Update call statuses (also runs via Celery Beat)

## 🚀 Deployment

### Production Checklist

1. **Environment Variables**
   - Set `DEBUG=False`
   - Use strong `SECRET_KEY`
   - Configure production database
   - Set up proper `ALLOWED_HOSTS`
   - Use HTTPS URLs for webhooks

2. **Database**
   - Use PostgreSQL in production
   - Set up database backups
   - Configure connection pooling

3. **Static Files**
   ```bash
   python manage.py collectstatic
   ```

4. **Celery**
   - Use Redis or RabbitMQ as broker
   - Set up Celery workers
   - Set up Celery Beat for periodic tasks
   - Configure monitoring

5. **Webhooks**
   - Use HTTPS URLs for webhooks
   - Set up proper authentication if needed
   - Monitor webhook delivery

6. **Security**
   - Review and update CORS settings
   - Enable authentication for API endpoints (currently disabled for testing)
   - Use environment variables for all secrets
   - Regularly update dependencies

### Docker Deployment

```dockerfile
FROM python:3.10-slim

WORKDIR /app
COPY requirements.txt .
RUN pip install --no-cache-dir -r requirements.txt

COPY . .
RUN python manage.py collectstatic --noinput

EXPOSE 8000
CMD ["gunicorn", "senior_engagement_app.wsgi:application", "--bind", "0.0.0.0:8000"]
```

## 📊 Monitoring

### Key Metrics to Monitor

- Call success rate
- Average call duration
- Conversation sentiment trends
- Memory extraction accuracy
- Webhook delivery success
- System performance metrics
- Celery task queue length

### Logging

The application logs important events:
- Call initiation and completion
- Webhook processing
- Error conditions
- Performance metrics
- Celery task execution

## 🔄 Celery Tasks

The application uses Celery for asynchronous task processing:

- `initiate_call_task` - Initiates calls via Twilio and Vapi.ai
- `process_vapi_webhook_task` - Processes webhooks from Vapi.ai
- `check_and_update_call_statuses` - Periodic task to update call statuses (runs every 10 seconds)

## 🎨 Admin Interface

The application uses Django Jazzmin for a beautiful admin interface:

- Modern, responsive design
- Custom icons for models
- Search functionality
- Easy navigation
- Access at `/admin/`

## 📦 Dependencies

- Django 4.2.7
- Django REST Framework 3.14.0
- Celery 5.3.4
- Redis 5.0.1
- Twilio 8.10.0
- psycopg2-binary 2.9.9
- django-jazzmin 2.6.0
- python-decouple 3.8

See `requirements.txt` for the complete list.

## 🤝 Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests
5. Submit a pull request

## 📄 License

This project is licensed under the MIT License - see the LICENSE file for details.

## 🆘 Support

For support and questions:
- Create an issue in the repository
- Check the documentation
- Review the API endpoints

## 🙏 Acknowledgments

- **Twilio** for reliable telephony infrastructure
- **Vapi.ai** for AI voice agent capabilities
- **Django** for the robust web framework
- **PostgreSQL** for reliable data storage
- **Celery** for asynchronous task processing
- **Jazzmin** for the beautiful admin interface

---

**Built with ❤️ for seniors and their families**
