from django.core.management.base import BaseCommand
from django.utils import timezone
from calls.models import Call, CallLog
from conversations.models import Conversation
from calls.tasks import process_completed_call_from_api
import requests
from django.conf import settings


class Command(BaseCommand):
    help = 'Process completed calls that are missing conversation data'

    def add_arguments(self, parser):
        parser.add_argument(
            '--dry-run',
            action='store_true',
            help='Show what would be processed without actually processing',
        )
        parser.add_argument(
            '--call-id',
            type=int,
            help='Process a specific call ID',
        )

    def handle(self, *args, **options):
        dry_run = options['dry_run']
        call_id = options.get('call_id')
        
        # Find completed calls without conversation data
        if call_id:
            calls = Call.objects.filter(id=call_id, twilio_status='completed')
        else:
            calls = Call.objects.filter(twilio_status='completed')
        
        calls_without_conversations = []
        for call in calls:
            if not Conversation.objects.filter(call=call).exists():
                calls_without_conversations.append(call)
        
        if not calls_without_conversations:
            self.stdout.write(
                self.style.SUCCESS('No completed calls without conversation data found.')
            )
            return
        
        self.stdout.write(f'Found {len(calls_without_conversations)} completed calls without conversation data:')
        
        headers = {
            'Authorization': f'Bearer {settings.VAPI_API_KEY}',
            'Content-Type': 'application/json'
        }
        
        processed_count = 0
        
        for call in calls_without_conversations:
            self.stdout.write(f'  - Call {call.id}: {call.senior.name} - VAPI ID: {call.vapi_call_id}')
            
            if not call.vapi_call_id:
                self.stdout.write(f'    No VAPI Call ID - skipping')
                continue
            
            # Get call data from VAPI API
            try:
                response = requests.get(
                    f'https://api.vapi.ai/call/{call.vapi_call_id}',
                    headers=headers
                )
                
                if response.status_code == 200:
                    call_data = response.json()
                    vapi_status = call_data.get('status', 'unknown')
                    
                    self.stdout.write(f'    VAPI Status: {vapi_status}')
                    
                    if vapi_status in ['completed', 'ended']:
                        self.stdout.write(f'    Processing conversation data...')
                        
                        if not dry_run:
                            try:
                                process_completed_call_from_api(call, call_data)
                                processed_count += 1
                                self.stdout.write(f'    ✓ Conversation processed successfully')
                                
                                # Log the processing
                                CallLog.objects.create(
                                    call=call,
                                    event_type='conversation_processed',
                                    message=f'Conversation data processed via management command',
                                    data={'processed_at': timezone.now().isoformat()}
                                )
                                
                            except Exception as e:
                                self.stdout.write(f'    ✗ Error processing: {str(e)}')
                        else:
                            self.stdout.write(f'    Would process conversation data (dry run)')
                    else:
                        self.stdout.write(f'    Call not completed in VAPI - skipping')
                        
                else:
                    self.stdout.write(f'    VAPI API Error: {response.status_code} - {response.text}')
                    
            except Exception as e:
                self.stdout.write(f'    Error checking VAPI: {str(e)}')
        
        if dry_run:
            self.stdout.write(
                self.style.WARNING(f'Dry run mode - would process {processed_count} conversations')
            )
        else:
            self.stdout.write(
                self.style.SUCCESS(f'Successfully processed {processed_count} conversations')
            )

