from rest_framework import serializers
from seniors.serializers import SeniorSerializer
from .models import Call, CallLog, CallPrompt


class CallPromptSerializer(serializers.ModelSerializer):
    """Serializer for CallPrompt model"""
    
    class Meta:
        model = CallPrompt
        fields = [
            'id', 'name', 'description', 'prompt_template', 'is_active', 'is_default',
            'created_at', 'updated_at', 'created_by'
        ]
        read_only_fields = ['id', 'created_at', 'updated_at', 'created_by']


class CallPromptListSerializer(serializers.ModelSerializer):
    """Lightweight serializer for listing prompts"""
    
    class Meta:
        model = CallPrompt
        fields = ['id', 'name', 'description', 'is_active', 'is_default']


class CallLogSerializer(serializers.ModelSerializer):
    """Serializer for CallLog model"""
    
    class Meta:
        model = CallLog
        fields = [
            'id', 'event_type', 'message', 'data', 'created_at'
        ]
        read_only_fields = ['id', 'created_at']


class CallSerializer(serializers.ModelSerializer):
    """Serializer for Call model"""
    
    senior = SeniorSerializer(read_only=True)
    prompt = CallPromptListSerializer(read_only=True)
    formatted_duration = serializers.ReadOnlyField()
    is_successful = serializers.ReadOnlyField()
    
    class Meta:
        model = Call
        fields = [
            'id', 'senior', 'twilio_sid', 'twilio_status',
            'duration', 'formatted_duration', 'call_start_time',
            'call_end_time', 'vapi_call_id', 'vapi_status',
            'prompt', 'ai_prompt', 'call_purpose', 'is_successful',
            'created_at', 'updated_at', 'initiated_by'
        ]
        read_only_fields = [
            'id', 'twilio_sid', 'twilio_status', 'duration',
            'call_start_time', 'call_end_time', 'vapi_call_id',
            'vapi_status', 'created_at', 'updated_at'
        ]


class CallCreateSerializer(serializers.ModelSerializer):
    """Serializer for creating Call instances"""
    
    class Meta:
        model = Call
        fields = ['senior', 'call_purpose', 'ai_prompt']
    
    def validate_senior(self, value):
        """Validate that the senior belongs to the authenticated user"""
        if value.caregiver != self.context['request'].user:
            raise serializers.ValidationError(
                "You can only create calls for seniors you manage."
            )
        return value
