from django.contrib import admin
from django import forms
from .models import Conversation, Memory, ConversationInsight


class ConversationAdminForm(forms.ModelForm):
    """Custom form for Conversation admin with better text field display"""
    
    class Meta:
        model = Conversation
        fields = '__all__'
        widgets = {
            'transcript': forms.Textarea(attrs={'rows': 15, 'cols': 100, 'style': 'font-family: monospace; white-space: pre-wrap;'}),
            'summary': forms.Textarea(attrs={'rows': 10, 'cols': 100, 'style': 'white-space: pre-wrap;'}),
        }


@admin.register(Conversation)
class ConversationAdmin(admin.ModelAdmin):
    form = ConversationAdminForm
    list_display = ['call', 'senior_name', 'has_transcript', 'has_summary', 'sentiment_score', 'engagement_level', 'created_at']
    list_filter = ['engagement_level', 'created_at']
    search_fields = ['call__senior__name', 'summary', 'transcript']
    readonly_fields = ['created_at', 'updated_at', 'senior_name', 'call_duration']
    
    fieldsets = (
        ('Call Information', {
            'fields': ('call', 'senior_name', 'call_duration')
        }),
        ('Conversation Content', {
            'fields': ('transcript', 'summary'),
            'description': 'Full transcript and AI-generated summary of the conversation'
        }),
        ('Analysis', {
            'fields': ('topics_discussed', 'key_memories', 'emotions_detected', 'sentiment_score', 'engagement_level')
        }),
        ('Follow-up', {
            'fields': ('follow_up_topics', 'follow_up_questions')
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    def senior_name(self, obj):
        """Display senior name"""
        return obj.call.senior.name if obj.call and obj.call.senior else 'N/A'
    senior_name.short_description = 'Senior'
    
    def call_duration(self, obj):
        """Display call duration"""
        if obj.call and obj.call.duration:
            minutes = obj.call.duration // 60
            seconds = obj.call.duration % 60
            return f"{minutes}m {seconds}s"
        return 'N/A'
    call_duration.short_description = 'Duration'
    
    def has_transcript(self, obj):
        """Check if transcript exists"""
        return bool(obj.transcript and obj.transcript.strip())
    has_transcript.short_description = 'Has Transcript'
    has_transcript.boolean = True
    
    def has_summary(self, obj):
        """Check if summary exists"""
        return bool(obj.summary and obj.summary.strip())
    has_summary.short_description = 'Has Summary'
    has_summary.boolean = True


@admin.register(Memory)
class MemoryAdmin(admin.ModelAdmin):
    list_display = ['title', 'senior', 'memory_type', 'importance_score', 'times_referenced', 'created_at']
    list_filter = ['memory_type', 'created_at', 'importance_score']
    search_fields = ['title', 'content', 'senior__name']
    readonly_fields = ['times_referenced', 'last_referenced', 'created_at', 'updated_at']
    
    fieldsets = (
        ('Memory Information', {
            'fields': ('senior', 'conversation', 'memory_type', 'title', 'content')
        }),
        ('Metadata', {
            'fields': ('importance_score', 'tags')
        }),
        ('Usage Tracking', {
            'fields': ('times_referenced', 'last_referenced')
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )


@admin.register(ConversationInsight)
class ConversationInsightAdmin(admin.ModelAdmin):
    list_display = ['title', 'senior', 'insight_type', 'confidence_score', 'created_at']
    list_filter = ['insight_type', 'created_at', 'confidence_score']
    search_fields = ['title', 'description', 'senior__name']
    readonly_fields = ['created_at', 'updated_at']
    
    fieldsets = (
        ('Insight Information', {
            'fields': ('senior', 'insight_type', 'title', 'description')
        }),
        ('Analysis Data', {
            'fields': ('data', 'confidence_score')
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )