import stripe

def check_subscription(subscription_id, supabase, price_id, quantity=1, team_id=""):
    if subscription_id is None:
        return {
            "status": "not_initialized",
            "seats": 0,
            "trial_ends": None,
            "checkout_link": "",
            "show_popup": True,
            "popup_reason": "CUSTOMER NEEDS TO BE REDIRECTED TO PAYMENT PAGE"
        }

    # Retrieve the original subscription
    original_subscription = stripe.Subscription.retrieve(subscription_id)
    customer_id = original_subscription["customer"]

    # Retrieve all subscriptions for the customer and find the newest one
    all_subscriptions = stripe.Subscription.list(customer=customer_id)
    newest_subscription = original_subscription
    for sub in all_subscriptions:
        if sub["created"] > newest_subscription["created"]:
            newest_subscription = sub

    # Use the newest subscription for all following checks
    subscription = newest_subscription
    status = subscription["status"]
    seats_amt = subscription.get("quantity", 0)
    trial_started = subscription.get("trial_end")
    payment_method_attached = stripe.Customer.retrieve(subscription["customer"])["invoice_settings"].get("default_payment_method", False)
    portal_session = stripe.billing_portal.Session.create(customer=subscription["customer"])
    portal = portal_session.url

    show_popup = False
    popup_reason = "No issues detected, subscription is active and within seat limits."

    if status in ["paused", "canceled", "past_due", "unpaid", "incomplete", "incomplete_expired"]:
        show_popup = True
        popup_reason = "Subscription is paused or canceled."
        checkout_session = stripe.checkout.Session.create(
                payment_method_types=['card'],
                customer=customer_id,
                line_items=[{
                    'price': price_id,  
                    'quantity': quantity
                }],
                mode='subscription',
                subscription_data={
                    'trial_end': None, 
                },
                automatic_tax={"enabled": True},
                metadata={
                    'subscription_id': subscription_id  
                },
                success_url='https://dashboard.rainconnect.de/dashboard',
                cancel_url='https://dashboard.rainconnect.de/dashboard' 
            )
        portal = checkout_session.url
    elif status == "trialing" and not payment_method_attached:
        show_popup = True
        popup_reason = "Trial period active but no payment method on file."

    # Update the TeamTable in Supabase if a newer subscription was found
    if subscription["id"] != original_subscription["id"]:
        supabase.table("TeamTable").update({"SubscriptionID": subscription["id"]}).eq("TeamID", team_id).execute()

    teamdata, count = supabase.table("TeamMembersTable").select("*").eq("TeamID", team_id).neq("IsRequesting", True).neq("Rejected", True).execute()
    active_members = len(teamdata[1])
    if active_members > seats_amt:
        show_popup = True
        popup_reason = "More active team members than licensed seats available."

    return {
        "status": status,
        "seats": seats_amt,
        "trial_ends": trial_started,
        "checkout_link": portal,
        "show_popup": show_popup,
        "popup_reason": popup_reason
    }
