from rest_framework import serializers

from apps.artiste.models import Album, Artiste, AudioMedia, Links,BankAccount
from apps.comments.models import MediaComments
from profiles.utils import song_duration


class LinkSerializer(serializers.ModelSerializer):
    link_type = serializers.CharField(source='get_link_type_display')

    class Meta:
        model = Links
        fields = ['link_url', 'link_type']


class CommentsSerializer(serializers.ModelSerializer):
    fan_id = serializers.CharField(source='fan.id')
    fan_name = serializers.CharField(source='fan.user.first_name')

    class Meta:
        model = MediaComments
        fields = ['id', 'comment', 'fan_name', 'fan_id', 'created_datetime']


class AudioMediaSerializer(serializers.ModelSerializer):
    album = serializers.CharField(source='album.album_name', read_only=True)
    media_comments = serializers.SerializerMethodField()
    media_comments_count = serializers.IntegerField(source='comments.count', read_only=True)
    likes_count = serializers.IntegerField(source='get_likes_count', read_only=True)
    dislikes_count = serializers.IntegerField(source='get_dislikes_count', read_only=True)
    score = serializers.IntegerField(source='get_score', read_only=True)

    class Meta:
        model = AudioMedia
        fields = [
            'id',
            'title',
            'score',
            'audio_file',
            'cover_image',
            'duration',
            'album',
            'likes_count',
            'dislikes_count',
            'media_comments_count',
            'media_comments',
        ]
        read_only_fields = ['cover_image']

    def get_media_comments(self, obj):
        return CommentsSerializer(obj.comments.all(), many=True).data

    def create(self, validated_data):
        try:
            file_duration = song_duration(validated_data['audio_file'])
        except Exception:
            file_duration = ""

        return AudioMedia.objects.create(
            title=validated_data['title'],
            audio_file=validated_data['audio_file'],
            artiste=self.context['artiste'],
            album=self.context['album_name'],
            cover_image=self.context['cover_image'],
            duration=file_duration,
        )


class ArtisteSerializer(serializers.ModelSerializer):
    links = LinkSerializer(many=True)
    audio_medias = AudioMediaSerializer(many=True)
    profile_song = serializers.SerializerMethodField()
    followers_count = serializers.IntegerField(source='followers.count')
    songs_count = serializers.IntegerField(source='get_audio_count')
    videos_count = serializers.IntegerField(source='get_video_count')
    albums_count = serializers.IntegerField(source='get_albums_count')
    artiste_comments_count = serializers.IntegerField(source='get_comments_count')
    is_following = serializers.SerializerMethodField()

    class Meta:
        model = Artiste
        fields = [
            'id',
            'stage_name',
            'profile_picture',
            'profile_song',
            'is_following',
            'followers_count',
            'songs_count',
            'videos_count',
            'albums_count',
            'artiste_comments_count',
            'links',
            'audio_medias',
            'verification_status'
        ]

    def get_profile_song(self, obj):
        data = AudioMediaSerializer(obj.profile_song).data

        return {
            'media_uuid': data.get('id', None),
            'title': data.get('title', None),
            'score': data.get('score', None),
            'audio_file': data.get('audio_file', None),
            'cover_image': data.get('cover_image', None),
            'media_comments_count': data.get('media_comments_count', None),
            'likes_count': data.get('likes_count', None),
            'dislikes_count': data.get('dislikes_count', None),
        }

    def get_is_following(self, obj):
        return obj.followers.filter(fan=self.context['request'].user.fan.id).exists()


class AlbumsSerializer(serializers.ModelSerializer):
    songs_count = serializers.IntegerField(source='get_songs_count', read_only=True)

    class Meta:
        model = Album
        fields = '__all__'
        read_only = ['artiste', 'songs_count']



class BankAccountSerializer(serializers.ModelSerializer):
    class Meta:
        model = BankAccount
        fields = [
            'account_holder_name',
            'bank_name',
            'account_number',
            'routing_number',
            'account_type',
            'currency'
        ]