import uuid

from django.db import models

from apps.actions.constants import ACTION_DISLIKE, ACTION_LIKE
from apps.artiste.constants import (
    PENDING,
    PLATFORM_CHOICES,
    PLATFORM_HIPHOP,
    VERIFICATION_STATUS,
    VERIFIED,
)
from apps.lib.base_model import BaseAbstractModel


class Links(BaseAbstractModel):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    link_url = models.URLField(blank=True)
    link_type = models.IntegerField(choices=PLATFORM_CHOICES, default=PLATFORM_HIPHOP)
    artiste = models.ForeignKey('artiste.Artiste', related_name='links', on_delete=models.CASCADE)

    class Meta:
        unique_together = ('artiste', 'link_type')

    def __str__(self):
        return f'LINK: {self.get_link_type_display()} - {self.artiste}'


class Artiste(BaseAbstractModel):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    user = models.OneToOneField('users.User', on_delete=models.CASCADE, related_name='artiste')
    stage_name = models.CharField(max_length=256, blank=True)
    profile_song = models.ForeignKey(
        'artiste.AudioMedia', on_delete=models.SET_NULL, related_name='artiste_profile_song', blank=True, null=True
    )
    fcm_token = models.CharField(max_length=244,null=True)
    device_type = models.CharField(max_length=244,null=True)
    arts = models.CharField(max_length=256, blank=True)
    profile_picture = models.ImageField(blank=True)
    is_accepted_terms = models.BooleanField(default=False)
    is_accepted_privacy = models.BooleanField(default=False)
    verification_status = models.IntegerField(default=PENDING, choices=VERIFICATION_STATUS, editable=False)

    def get_albums_count(self):
        return self.albums.count()

    def get_audio_count(self):
        return self.audio_medias.count()

    def get_video_count(self):
        return self.video_medias.count()

    def get_comments_count(self):
        return self.comments.count()

    @property
    def is_verified(self):
        return self.verification_status == VERIFIED

    def get_followers_count(self):
        return self.followers.filter().count()

    def get_followers(self):
        return self.followers.filter()

    def __str__(self) -> str:
        return self.stage_name


class BaseMedia(BaseAbstractModel):
    artiste = models.ForeignKey('artiste.Artiste', on_delete=models.DO_NOTHING, related_name='medias')
    title = models.CharField(max_length=512, blank=False, null=True)

    class Meta:
        abstract = True


class Album(BaseAbstractModel):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    album_name = models.CharField(max_length=512, blank=False, null=True)
    artiste = models.ForeignKey(
        'artiste.Artiste', on_delete=models.CASCADE, related_name='albums', null=True, blank=True
    )

    class Meta:
        verbose_name = 'Album'
        verbose_name_plural = 'Albums'
        ordering = ('-created_datetime',)

    def get_songs_count(self):
        return self.albums.count()

    def __str__(self) -> str:
        return f'{self.artiste.stage_name} - {self.album_name}'


class AudioMedia(BaseMedia):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    album = models.ForeignKey('artiste.Album', on_delete=models.PROTECT, related_name='albums', blank=True, null=True)
    artiste = models.ForeignKey('artiste.Artiste', on_delete=models.PROTECT, related_name='audio_medias', null=True)
    cover_image = models.ImageField(upload_to="cover-images", blank=True)
    duration = models.CharField(max_length=15, blank=True, editable=False)
    audio_file = models.FileField()
    score = models.FloatField(default=0)

    def get_likes_count(self):
        return self.likes.filter(action_type=ACTION_LIKE).count()

    def get_dislikes_count(self):
        return self.likes.filter(action_type=ACTION_DISLIKE).count()

    def get_score(self):
        return int(self.score)

    def __str__(self):
        return f'{self.artiste.stage_name} - {self.title}'


class VideoMedia(BaseMedia):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    artiste = models.ForeignKey('artiste.Artiste', on_delete=models.CASCADE, related_name='video_medias')
    cover_image = models.ImageField(upload_to="cover-images", blank=True)
    duration = models.CharField(max_length=15, blank=True, editable=False)
    video_file = models.FileField()

    def __str__(self):
        return f'{self.artiste.stage_name} - {self.title}'


class BankAccount(models.Model):
    user = models.OneToOneField('users.User', on_delete=models.CASCADE, related_name='user')
    account_holder_name = models.CharField(max_length=100, null=True)
    bank_name = models.CharField(max_length=100, null=True)
    account_number = models.CharField(max_length=100, null=True)
    routing_number = models.CharField(max_length=100, null=True)
    account_type = models.CharField(max_length=100, null=True)
    currency = models.CharField(max_length=100, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    end_date = models.DateTimeField(null=True) 
    
    class Meta:
        db_table = 'bank_account'


    


    
    
    