from re import M

from django.contrib.auth import get_user_model
from rest_framework import status
from rest_framework.generics import GenericAPIView
from rest_framework.response import Response

from apps.artiste.models import Artiste, AudioMedia
from apps.comments.api.v1.serializers import (
    ArtisteCommentsSerializer,
    DeleteArtisteCommentsSerializer,
    DeleteMediaCommentsSerializer,
    MediaCommentsSerializer,
)
from apps.comments.models import ArtisteComments, MediaComments
from apps.comments.pagination import CustomPagination
from apps.lib.utils import GetObjectMixin

User = get_user_model()


class MediaCommentsAPIView(GetObjectMixin, GenericAPIView):
    queryset = MediaComments.objects.all()
    serializer_class = MediaCommentsSerializer
    pagination_class = CustomPagination

    def get(self, request, media_uuid):
        try:
            media_comments = MediaComments.objects.filter(media=media_uuid)
        except MediaComments.DoesNotExist:
            return Response({"error": "Media does not exist"}, status=status.HTTP_404_NOT_FOUND)

        page = self.paginate_queryset(media_comments)

        if page is not None:
            serializer = self.get_serializer(page, many=True)
            return self.get_paginated_response(serializer.data)

        serializer = self.get_serializer(media_comments, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)


class CreateMediaCommentAPIView(GetObjectMixin, GenericAPIView):
    queryset = MediaComments.objects.all()
    serializer_class = MediaCommentsSerializer

    def post(self, request, media_uuid):
        media = self.get_object_by_model(AudioMedia, media_uuid)
        fan = self.get_related_object_by_model(User, self.request.user.id, 'fan')

        serializer = self.get_serializer(data=request.data, context={'fan': fan, 'media': media})
        serializer.is_valid(raise_exception=True)
        serializer.save()

        return Response(serializer.data, status=status.HTTP_201_CREATED)


class DeleteMediaCommentAPIView(GetObjectMixin, GenericAPIView):
    queryset = MediaComments.objects.all()
    serializer_class = DeleteMediaCommentsSerializer

    def delete(self, request, comment_uuid):
        try:
            comment = self.get_object_by_model(MediaComments, comment_uuid)
        except MediaComments.DoesNotExist:
            return Response({"error": "Comment does not exist"}, status=status.HTTP_404_NOT_FOUND)

        serializer = self.get_serializer(instance=comment, data=request.data)
        serializer.is_valid(raise_exception=True)
        serializer.save()

        return Response(serializer.data, status=status.HTTP_200_OK)


class ArtisteCommentsAPIView(GetObjectMixin, GenericAPIView):
    queryset = ArtisteComments.objects.all()
    serializer_class = ArtisteCommentsSerializer
    pagination_class = CustomPagination

    def get(self, request, artiste_uuid):
        try:
            artiste_comments = ArtisteComments.objects.filter(artiste=artiste_uuid)
        except ArtisteComments.DoesNotExist:
            return Response({"error": "ArtisteComments does not exist"}, status=status.HTTP_404_NOT_FOUND)

        page = self.paginate_queryset(artiste_comments)

        if page is not None:
            serializer = self.get_serializer(page, many=True)
            return self.get_paginated_response(serializer.data)

        serializer = self.get_serializer(artiste_comments, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)


class CreateArtisteCommentAPIView(GetObjectMixin, GenericAPIView):
    queryset = ArtisteComments.objects.all()
    serializer_class = ArtisteCommentsSerializer

    def post(self, request, artiste_uuid):
        artiste = self.get_object_by_model(Artiste, artiste_uuid)
        fan = self.get_related_object_by_model(User, self.request.user.id, 'fan')

        serializer = self.get_serializer(data=request.data, context={'fan': fan, 'artiste': artiste})
        serializer.is_valid(raise_exception=True)
        serializer.save()

        return Response(serializer.data, status=status.HTTP_201_CREATED)


class DeleteArtisteCommentAPIView(GetObjectMixin, GenericAPIView):
    queryset = ArtisteComments.objects.all()
    serializer_class = DeleteArtisteCommentsSerializer

    def delete(self, request, comment_uuid):
        try:
            comment = self.get_object_by_model(ArtisteComments, comment_uuid)
        except MediaComments.DoesNotExist:
            return Response({"error": "ArtisteComment does not exist"}, status=status.HTTP_404_NOT_FOUND)

        serializer = self.get_serializer(instance=comment, data=request.data)
        serializer.is_valid(raise_exception=True)
        serializer.save()

        return Response(serializer.data, status=status.HTTP_200_OK)
