from flask import Flask, jsonify, request
from fetch import EmailChecker, setup_logging
from email_poller import EmailPoller
from email_sender import EmailSender
import logging
from typing import Dict, List
import os
from dotenv import load_dotenv
import threading

# Load environment variables from .env file
load_dotenv()

app = Flask(__name__)
logger = setup_logging("api.log")

# Configuration - Load from environment variables or use defaults
# EMAIL_ADDRESS = os.getenv("EMAIL_ADDRESS", "developer@esferasoft.com")
EMAIL_ADDRESS = "info@troyrides.com"
CLIENT_ID = "cc46473e-a7d7-4f74-8045-97f57672db79"
CLIENT_SECRET = "0eB8Q~zpWLgBaBjbuSEdCcWw0PqebXYIZH8DMcfh"  # Should be set via environment variable or .env file
TENANT_ID = "0b737fe7-8092-4deb-8adc-2c25370f9d55"
DEFAULT_SENDER = os.getenv("DEFAULT_SENDER", "scheduling@usetwine.com")
DEFAULT_FOLDER = os.getenv("DEFAULT_FOLDER", "inbox")
DEFAULT_FROM_DATE = os.getenv("DEFAULT_FROM_DATE", "2025-12-17")  # Default date filter to match fetch.py
TEST_EMAIL_OVERRIDE = os.getenv("TEST_EMAIL_OVERRIDE", "developer@esferasoft.com")  # Override extracted email for testing

# Polling configuration
POLLING_INTERVAL = int(os.getenv("POLLING_INTERVAL", 60))  # Default: 60 seconds
POLLING_AUTO_START = os.getenv("POLLING_AUTO_START", "true").lower() == "true"
POLLING_ENABLED = os.getenv("POLLING_ENABLED", "true").lower() == "true"
TRACKING_FILE = os.getenv("TRACKING_FILE", "processed_emails.json")
# Callback URL - set to your server's callback endpoint or external webhook
# Default: Uses internal callback endpoint
CALLBACK_URL = os.getenv("CALLBACK_URL", None)  # Set to None to use internal callback, or provide external URL
USE_INTERNAL_CALLBACK = os.getenv("USE_INTERNAL_CALLBACK", "true").lower() == "true"  # Use internal callback by default

# Email sending configuration
MAIL_HOST = os.getenv("MAIL_HOST", "smtp.gmail.com")
MAIL_PORT = int(os.getenv("MAIL_PORT", 465))
MAIL_USERNAME = os.getenv("MAIL_USERNAME", "testermohali@gmail.com")
MAIL_PASSWORD = os.getenv("MAIL_PASSWORD", "duxfjduhrfiizcav")
MAIL_ENCRYPTION = os.getenv("MAIL_ENCRYPTION", "ssl")
MAIL_FROM_ADDRESS = os.getenv("MAIL_FROM_ADDRESS", "testermohali@gmail.com")
MAIL_FROM_NAME = os.getenv("MAIL_FROM_NAME", "Troy Rides")

# Global email checker instance (will be initialized on first request)
email_checker = None

# Global email poller instance
email_poller: EmailPoller = None

# Global email sender instance
email_sender: EmailSender = None

def reset_email_checker():
    """Reset the global email checker instance (useful when token expires)"""
    global email_checker
    email_checker = None
    logger.info("Email checker instance has been reset")


def get_email_sender() -> EmailSender:
    """Get or create email sender instance"""
    global email_sender
    if email_sender is None:
        email_sender = EmailSender(
            host=MAIL_HOST,
            port=MAIL_PORT,
            username=MAIL_USERNAME,
            password=MAIL_PASSWORD,
            encryption=MAIL_ENCRYPTION,
            from_address=MAIL_FROM_ADDRESS,
            from_name=MAIL_FROM_NAME
        )
        logger.info("Email sender instance created")
    return email_sender


def get_email_poller() -> EmailPoller:
    """Get or create email poller instance"""
    global email_poller
    if email_poller is None:
        checker = get_email_checker()
        
        # Determine callback URL
        callback_url = CALLBACK_URL
        if not callback_url and USE_INTERNAL_CALLBACK:
            # Use internal callback endpoint
            # Get the server URL from environment or construct it
            server_host = os.getenv('HOST', '0.0.0.0')
            server_port = os.getenv('PORT', '5000')
            # For internal callbacks, use localhost
            if server_host == '0.0.0.0':
                callback_url = f"http://localhost:{server_port}/api/polling/callback"
            else:
                callback_url = f"http://{server_host}:{server_port}/api/polling/callback"
            logger.info(f"Using internal callback URL: {callback_url}")
        elif callback_url:
            logger.info(f"Using external callback URL: {callback_url}")
        else:
            logger.info("No callback URL configured - notifications will only be logged")
        
        email_poller = EmailPoller(
            email_checker=checker,
            sender_email=DEFAULT_SENDER,
            folder=DEFAULT_FOLDER,
            interval=POLLING_INTERVAL,
            tracking_file=TRACKING_FILE,
            callback_url=callback_url
        )
        logger.info("Email poller instance created")
    return email_poller


def get_email_checker():
    """Get or create email checker instance"""
    global email_checker
    if email_checker is None:
        if not CLIENT_SECRET:
            raise ValueError("CLIENT_SECRET environment variable is not set")
        logger.info(f"Initializing EmailChecker for: {EMAIL_ADDRESS}")
        logger.info(f"Using CLIENT_ID: {CLIENT_ID}")
        logger.info(f"Using TENANT_ID: {TENANT_ID}")
        email_checker = EmailChecker(
            email_address=EMAIL_ADDRESS,
            client_id=CLIENT_ID,
            client_secret=CLIENT_SECRET,
            tenant_id=TENANT_ID
        )
        
        # Connect to Microsoft Graph API
        logger.info("Attempting to connect to Microsoft Graph API...")
        connection_result = email_checker.connect()
        logger.info(f"Connection result: {connection_result}")
        if not connection_result:
            logger.error("Failed to authenticate with Microsoft Graph API")
            raise Exception("Failed to authenticate with Microsoft Graph API")
        logger.info("Successfully authenticated with Microsoft Graph API")
        logger.info(f"Access token obtained: {'Yes' if email_checker.access_token else 'No'}")
    else:
        logger.info("Reusing existing email checker instance")
        logger.info(f"Access token status: {'Has token' if email_checker.access_token else 'No token'}")
        # Re-authenticate if token is missing
        if not email_checker.access_token:
            logger.warning("Access token is missing, re-authenticating...")
            connection_result = email_checker.connect()
            if not connection_result:
                logger.error("Failed to re-authenticate with Microsoft Graph API")
                # Reset the instance so it will be recreated next time
                email_checker = None
                raise Exception("Failed to re-authenticate with Microsoft Graph API")
            logger.info("Successfully re-authenticated")
    
    return email_checker


@app.route('/health', methods=['GET'])
def health_check():
    """Health check endpoint"""
    return jsonify({
        "status": "healthy",
        "service": "Troy Email Fetcher API"
    }), 200


@app.route('/api/debug', methods=['GET'])
def debug_info():
    """Debug endpoint to check email checker status"""
    try:
        checker = get_email_checker()
        return jsonify({
            "success": True,
            "email_address": EMAIL_ADDRESS,
            "has_access_token": checker.access_token is not None,
            "access_token_length": len(checker.access_token) if checker.access_token else 0,
            "default_sender": DEFAULT_SENDER,
            "default_from_date": DEFAULT_FROM_DATE,
            "client_id": CLIENT_ID,
            "tenant_id": TENANT_ID
        }), 200
    except Exception as e:
        logger.error(f"Debug endpoint error: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/emails', methods=['GET'])
def fetch_emails():
    """
    Fetch emails from the configured sender
    
    Query Parameters:
    - sender: Email address to filter by (default: scheduling@usetwine.com)
    - folder: Email folder to search (default: inbox)
    - unread_only: Only fetch unread emails (default: false)
    - from_date: Fetch emails from this date onwards (format: YYYY-MM-DD)
    - limit: Maximum number of emails to return (default: 100)
    
    Returns:
    - JSON array of email objects with extracted booking details
    """
    try:
        # Get query parameters
        sender = request.args.get('sender', DEFAULT_SENDER)
        folder = request.args.get('folder', DEFAULT_FOLDER)
        unread_only = request.args.get('unread_only', 'false').lower() == 'true'
        from_date = request.args.get('from_date', DEFAULT_FROM_DATE)  # Use default date if not provided
        limit = int(request.args.get('limit', 100))
        
        logger.info(f"API Request - sender: {sender}, folder: {folder}, unread_only: {unread_only}, from_date: {from_date}, limit: {limit}")
        
        # Get email checker instance
        try:
            checker = get_email_checker()
        except Exception as e:
            logger.error(f"Failed to get email checker: {e}", exc_info=True)
            return jsonify({
                "success": False,
                "error": f"Failed to initialize email checker: {str(e)}"
            }), 500
        
        # Fetch emails
        logger.info("Calling check_emails_from_sender...")
        logger.info(f"Parameters: sender_email={sender}, folder={folder}, unread_only={unread_only}, top={limit}, from_date={from_date}")
        try:
            emails = checker.check_emails_from_sender(
                sender_email=sender,
                folder=folder,
                unread_only=unread_only,
                top=limit,
                from_date=from_date
            )
            logger.info(f"check_emails_from_sender returned {len(emails)} emails")
            if emails:
                logger.info(f"First email ID: {emails[0].get('id', 'N/A')}")
                logger.info(f"First email subject: {emails[0].get('subject', 'N/A')}")
            else:
                logger.warning("check_emails_from_sender returned an empty list!")
                logger.warning("This might indicate:")
                logger.warning("1. No emails match the filter criteria")
                logger.warning("2. Authentication issue")
                logger.warning("3. Email checker access_token status: " + ("Has token" if checker.access_token else "No token"))
                # If we got 0 emails and have a token, it might be expired - try resetting and retrying once
                if checker.access_token:
                    logger.warning("Token exists but got 0 emails. Token may be expired. Resetting email checker and retrying...")
                    reset_email_checker()
                    checker = get_email_checker()
                    logger.info("Retrying email fetch after reset...")
                    emails = checker.check_emails_from_sender(
                        sender_email=sender,
                        folder=folder,
                        unread_only=unread_only,
                        top=limit,
                        from_date=from_date
                    )
                    logger.info(f"After reset, check_emails_from_sender returned {len(emails)} emails")
        except Exception as e:
            logger.error(f"Error in check_emails_from_sender: {e}", exc_info=True)
            # If it's an authentication error, reset and retry once
            if "401" in str(e) or "Unauthorized" in str(e) or "expired" in str(e).lower():
                logger.warning("Detected authentication error. Resetting email checker and retrying...")
                reset_email_checker()
                try:
                    checker = get_email_checker()
                    emails = checker.check_emails_from_sender(
                        sender_email=sender,
                        folder=folder,
                        unread_only=unread_only,
                        top=limit,
                        from_date=from_date
                    )
                    logger.info(f"After reset and retry, check_emails_from_sender returned {len(emails)} emails")
                except Exception as retry_e:
                    logger.error(f"Error on retry after reset: {retry_e}", exc_info=True)
                    return jsonify({
                        "success": False,
                        "error": f"Failed to fetch emails after retry: {str(retry_e)}"
                    }), 500
            else:
                return jsonify({
                    "success": False,
                    "error": f"Failed to fetch emails: {str(e)}"
                }), 500
        
        # Override extracted email for testing
        logger.info(f"Processing {len(emails)} emails for response...")
        for email in emails:
            if "extracted_details" in email and email["extracted_details"]:
                email["extracted_details"]["email"] = TEST_EMAIL_OVERRIDE
        
        # Format response
        response_data = {
            "success": True,
            "count": len(emails),
            "emails": emails
        }
        
        logger.info(f"API Response - Successfully fetched {len(emails)} emails")
        logger.info(f"Response data structure: success={response_data['success']}, count={response_data['count']}, emails list length={len(response_data['emails'])}")
        return jsonify(response_data), 200
        
    except ValueError as e:
        logger.error(f"API Error - Configuration error: {e}")
        return jsonify({
            "success": False,
            "error": str(e)
        }), 400
        
    except Exception as e:
        logger.error(f"API Error - Failed to fetch emails: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/emails/summary', methods=['GET'])
def fetch_emails_summary():
    """
    Fetch emails and return only a summary (without full body)
    
    Same query parameters as /api/emails
    Returns only essential fields for each email
    """
    try:
        # Get query parameters
        sender = request.args.get('sender', DEFAULT_SENDER)
        folder = request.args.get('folder', DEFAULT_FOLDER)
        unread_only = request.args.get('unread_only', 'false').lower() == 'true'
        from_date = request.args.get('from_date', DEFAULT_FROM_DATE)  # Use default date if not provided
        limit = int(request.args.get('limit', 100))
        
        logger.info(f"API Request (Summary) - sender: {sender}, folder: {folder}, from_date: {from_date}")
        
        # Get email checker instance
        try:
            checker = get_email_checker()
        except Exception as e:
            logger.error(f"Failed to get email checker: {e}", exc_info=True)
            return jsonify({
                "success": False,
                "error": f"Failed to initialize email checker: {str(e)}"
            }), 500
        
        # Fetch emails
        try:
            emails = checker.check_emails_from_sender(
                sender_email=sender,
                folder=folder,
                unread_only=unread_only,
                top=limit,
                from_date=from_date
            )
            logger.info(f"check_emails_from_sender returned {len(emails)} emails")
        except Exception as e:
            logger.error(f"Error in check_emails_from_sender: {e}", exc_info=True)
            return jsonify({
                "success": False,
                "error": f"Failed to fetch emails: {str(e)}"
            }), 500
        
        # Override extracted email for testing
        for email in emails:
            if "extracted_details" in email and email["extracted_details"]:
                email["extracted_details"]["email"] = TEST_EMAIL_OVERRIDE
        
        # Create summary (exclude full body and raw_body)
        summaries = []
        for email in emails:
            summary = {
                "id": email.get("id"),
                "subject": email.get("subject"),
                "from": email.get("from"),
                "date": email.get("date"),
                "extracted_details": email.get("extracted_details", {})
            }
            # Remove raw_body from extracted_details if present
            if "raw_body" in summary["extracted_details"]:
                del summary["extracted_details"]["raw_body"]
            summaries.append(summary)
        
        response_data = {
            "success": True,
            "count": len(summaries),
            "emails": summaries
        }
        
        logger.info(f"API Response (Summary) - Successfully fetched {len(summaries)} email summaries")
        return jsonify(response_data), 200
        
    except ValueError as e:
        logger.error(f"API Error - Configuration error: {e}")
        return jsonify({
            "success": False,
            "error": str(e)
        }), 400
        
    except Exception as e:
        logger.error(f"API Error - Failed to fetch emails: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/emails/<email_id>', methods=['GET'])
def get_email_by_id(email_id):
    """
    Get a specific email by ID
    
    Returns the full email details including body
    """
    try:
        logger.info(f"API Request - Get email by ID: {email_id}")
        
        # Get email checker instance
        checker = get_email_checker()
        
        # Fetch the specific email
        # Note: This requires fetching from the mailbox and finding the email
        # For now, we'll search for it
        emails = checker.check_emails_from_sender(
            sender_email=DEFAULT_SENDER,
            folder=DEFAULT_FOLDER,
            unread_only=False,
            top=1000,  # Increase limit to find the email
            from_date=None
        )
        
        # Find the email by ID
        email = None
        for e in emails:
            if e.get("id") == email_id:
                email = e
                break
        
        if not email:
            return jsonify({
                "success": False,
                "error": f"Email with ID {email_id} not found"
            }), 404
        
        # Override extracted email for testing
        if "extracted_details" in email and email["extracted_details"]:
            email["extracted_details"]["email"] = TEST_EMAIL_OVERRIDE
        
        response_data = {
            "success": True,
            "email": email
        }
        
        logger.info(f"API Response - Found email: {email_id}")
        return jsonify(response_data), 200
        
    except Exception as e:
        logger.error(f"API Error - Failed to get email: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/polling/test', methods=['GET'])
def test_polling():
    """Test endpoint to verify polling routes are loaded"""
    return jsonify({
        "success": True,
        "message": "Polling routes are loaded",
        "polling_enabled": POLLING_ENABLED
    }), 200


@app.route('/api/polling/start', methods=['POST'])
def start_polling():
    """
    Start the email polling service
    
    Query Parameters:
    - interval: Optional polling interval in seconds (default: configured interval)
    
    Returns:
    - JSON with polling status
    """
    try:
        if not POLLING_ENABLED:
            return jsonify({
                "success": False,
                "error": "Polling is disabled"
            }), 400
        
        # Get optional interval parameter
        interval = request.args.get('interval', type=int)
        
        poller = get_email_poller()
        
        # Update interval if provided
        if interval:
            if interval < 10:
                return jsonify({
                    "success": False,
                    "error": "Interval must be at least 10 seconds"
                }), 400
            poller.set_interval(interval)
        
        # Start polling
        if poller.start_polling():
            logger.info("Polling service started via API")
            return jsonify({
                "success": True,
                "message": "Polling started",
                "status": poller.get_status()
            }), 200
        else:
            return jsonify({
                "success": False,
                "error": "Polling is already running",
                "status": poller.get_status()
            }), 400
            
    except Exception as e:
        logger.error(f"Error starting polling: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/polling/stop', methods=['POST'])
def stop_polling():
    """
    Stop the email polling service
    
    Returns:
    - JSON with polling status
    """
    try:
        poller = get_email_poller()
        
        if poller.stop_polling():
            logger.info("Polling service stopped via API")
            return jsonify({
                "success": True,
                "message": "Polling stopped",
                "status": poller.get_status()
            }), 200
        else:
            return jsonify({
                "success": False,
                "error": "Polling is not running",
                "status": poller.get_status()
            }), 400
            
    except Exception as e:
        logger.error(f"Error stopping polling: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/polling/status', methods=['GET'])
def polling_status():
    """
    Get current polling service status
    
    Returns:
    - JSON with polling status information
    """
    try:
        poller = get_email_poller()
        status = poller.get_status()
        
        return jsonify({
            "success": True,
            "status": status
        }), 200
        
    except Exception as e:
        logger.error(f"Error getting polling status: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/polling/config', methods=['POST'])
def update_polling_config():
    """
    Update polling configuration
    
    JSON Body:
    - interval: Polling interval in seconds (optional, must be >= 10)
    
    Returns:
    - JSON with updated configuration
    """
    try:
        if not POLLING_ENABLED:
            return jsonify({
                "success": False,
                "error": "Polling is disabled"
            }), 400
        
        data = request.get_json() or {}
        interval = data.get('interval')
        
        poller = get_email_poller()
        
        if interval is not None:
            if not isinstance(interval, int) or interval < 10:
                return jsonify({
                    "success": False,
                    "error": "Interval must be an integer >= 10 seconds"
                }), 400
            poller.set_interval(interval)
            logger.info(f"Polling interval updated to {interval} seconds via API")
        
        return jsonify({
            "success": True,
            "message": "Configuration updated",
            "status": poller.get_status()
        }), 200
        
    except Exception as e:
        logger.error(f"Error updating polling config: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/send-driver-info', methods=['POST'])
def send_driver_info():
    """
    Send driver information email with OTP to customer
    
    Request Body (JSON):
    {
        "email": "customer@example.com",
        "driver_name": "John Smith",
        "driver_email": "driver@troyrides.com",
        "driver_phone": "+1-555-123-4567",
        "otp": "123456"
    }
    
    Returns:
    - JSON with success status
    """
    try:
        data = request.get_json()
        
        if not data:
            return jsonify({
                "success": False,
                "error": "No data received"
            }), 400
        
        # Validate required fields
        email = data.get('email')
        driver_name = data.get('driver_name')
        driver_email = data.get('driver_email')
        driver_phone = data.get('driver_phone')
        otp = data.get('otp')
        
        if not email:
            return jsonify({
                "success": False,
                "error": "email is required"
            }), 400
        
        if not driver_name:
            return jsonify({
                "success": False,
                "error": "driver_name is required"
            }), 400
        
        if not driver_email:
            return jsonify({
                "success": False,
                "error": "driver_email is required"
            }), 400
        
        if not driver_phone:
            return jsonify({
                "success": False,
                "error": "driver_phone is required"
            }), 400
        
        if not otp:
            return jsonify({
                "success": False,
                "error": "otp is required"
            }), 400
        
        # Validate email format (basic)
        if '@' not in email or '.' not in email.split('@')[-1]:
            return jsonify({
                "success": False,
                "error": "Invalid recipient email format"
            }), 400
        
        if '@' not in driver_email or '.' not in driver_email.split('@')[-1]:
            return jsonify({
                "success": False,
                "error": "Invalid driver email format"
            }), 400
        
        logger.info(f"Sending driver info email to {email}")
        logger.info(f"Driver: {driver_name} ({driver_email}), Phone: {driver_phone}, OTP: {otp}")
        
        # Get email sender and send email
        sender = get_email_sender()
        success = sender.send_driver_info(
            recipient_email=email,
            driver_name=driver_name,
            driver_email=driver_email,
            driver_phone=driver_phone,
            otp=str(otp)
        )
        
        if success:
            logger.info(f"Driver info email sent successfully to {email}")
            return jsonify({
                "success": True,
                "message": "Driver information email sent successfully",
                "recipient": email,
                "driver_name": driver_name
            }), 200
        else:
            logger.error(f"Failed to send driver info email to {email}")
            return jsonify({
                "success": False,
                "error": "Failed to send email. Please check SMTP configuration and logs."
            }), 500
            
    except Exception as e:
        logger.error(f"Error sending driver info email: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/send-fare-estimate', methods=['POST'])
def send_fare_estimate():
    """
    Send fare estimate email to customer
    
    Request Body (JSON):
    {
        "email": "customer@example.com",
        "estimate_fare": "45.00",
        "payment_link": "https://payment.example.com/pay/12345",
        "customer_name": "John Doe"  // optional
    }
    
    Returns:
    - JSON with success status
    """
    try:
        data = request.get_json()
        
        if not data:
            return jsonify({
                "success": False,
                "error": "No data received"
            }), 400
        
        # Validate required fields
        email = data.get('email')
        estimate_fare = data.get('estimate_fare')
        payment_link = data.get('payment_link')
        customer_name = data.get('customer_name')  # Optional
        
        if not email:
            return jsonify({
                "success": False,
                "error": "Email is required"
            }), 400
        
        if not estimate_fare:
            return jsonify({
                "success": False,
                "error": "estimate_fare is required"
            }), 400
        
        if not payment_link:
            return jsonify({
                "success": False,
                "error": "payment_link is required"
            }), 400
        
        # Validate email format (basic)
        if '@' not in email or '.' not in email.split('@')[-1]:
            return jsonify({
                "success": False,
                "error": "Invalid email format"
            }), 400
        
        logger.info(f"Sending fare estimate email to {email}")
        logger.info(f"Fare: ${estimate_fare}, Payment Link: {payment_link}")
        
        # Get email sender and send email
        sender = get_email_sender()
        success = sender.send_fare_estimate(
            recipient_email=email,
            estimate_fare=str(estimate_fare),
            payment_link=payment_link,
            customer_name=customer_name
        )
        
        if success:
            logger.info(f"Fare estimate email sent successfully to {email}")
            return jsonify({
                "success": True,
                "message": "Fare estimate email sent successfully",
                "recipient": email,
                "estimate_fare": estimate_fare
            }), 200
        else:
            logger.error(f"Failed to send fare estimate email to {email}")
            return jsonify({
                "success": False,
                "error": "Failed to send email. Please check SMTP configuration and logs."
            }), 500
            
    except Exception as e:
        logger.error(f"Error sending fare estimate email: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.route('/api/polling/callback', methods=['POST'])
def email_notification_callback():
    """
    Callback endpoint to receive email notifications from the polling service
    
    This endpoint receives POST requests when new emails are processed.
    You can configure the polling service to call this endpoint by setting:
    CALLBACK_URL=http://your-server:5000/api/polling/callback
    
    Request Body (from polling service):
    {
        "email": {
            "id": "email_id",
            "subject": "...",
            "from": "...",
            "date": "...",
            "extracted_details": {...}
        },
        "timestamp": "2026-01-05T10:05:01",
        "source": "email_poller"
    }
    
    Returns:
    - JSON confirmation
    """
    try:
        data = request.get_json()
        
        if not data:
            return jsonify({
                "success": False,
                "error": "No data received"
            }), 400
        
        email_data = data.get('email', {})
        timestamp = data.get('timestamp', '')
        source = data.get('source', 'unknown')
        
        if not email_data:
            return jsonify({
                "success": False,
                "error": "No email data in request"
            }), 400
        
        email_id = email_data.get('id', 'unknown')
        subject = email_data.get('subject', 'No Subject')
        extracted_details = email_data.get('extracted_details', {})
        
        logger.info("="*60)
        logger.info("EMAIL NOTIFICATION RECEIVED")
        logger.info(f"Source: {source}")
        logger.info(f"Timestamp: {timestamp}")
        logger.info(f"Email ID: {email_id}")
        logger.info(f"Subject: {subject}")
        logger.info(f"From: {email_data.get('from', 'Unknown')}")
        logger.info(f"Date: {email_data.get('date', 'Unknown')}")
        
        if extracted_details:
            logger.info("Extracted Details:")
            logger.info(f"  Name: {extracted_details.get('name', 'N/A')}")
            logger.info(f"  Phone: {extracted_details.get('phone', 'N/A')}")
            logger.info(f"  Email: {extracted_details.get('email', 'N/A')}")
            logger.info(f"  Address: {extracted_details.get('address', 'N/A')}")
            if extracted_details.get('date_time'):
                logger.info(f"  Date & Time: {extracted_details.get('date_time', 'N/A')}")
            logger.info(f"  Pick Up Location: {extracted_details.get('pick_up_location', 'N/A')}")
            logger.info(f"  Drop Off Location: {extracted_details.get('drop_off_location', 'N/A')}")
            logger.info(f"  Number of Passengers: {extracted_details.get('number_of_passengers', 'N/A')}")
        
        logger.info("="*60)
        
        # Here you can add custom processing:
        # - Save to database
        # - Send to another service
        # - Trigger webhooks
        # - Send notifications (SMS, email, etc.)
        
        return jsonify({
            "success": True,
            "message": "Email notification received and processed",
            "email_id": email_id,
            "subject": subject,
            "timestamp": timestamp
        }), 200
        
    except Exception as e:
        logger.error(f"Error processing email notification callback: {e}", exc_info=True)
        return jsonify({
            "success": False,
            "error": str(e)
        }), 500


@app.errorhandler(404)
def not_found(error):
    return jsonify({
        "success": False,
        "error": "Endpoint not found"
    }), 404


@app.errorhandler(500)
def internal_error(error):
    return jsonify({
        "success": False,
        "error": "Internal server error"
    }), 500


if __name__ == '__main__':
    # Check if CLIENT_SECRET is set
    if not CLIENT_SECRET:
        logger.error("CLIENT_SECRET environment variable is not set!")
        logger.error("Please set it before running the API:")
        logger.error("export CLIENT_SECRET='your-secret-value'")
        exit(1)
    
    # Get port from environment or use default
    port = int(os.getenv('PORT', 5000))
    host = os.getenv('HOST', '0.0.0.0')
    
    logger.info(f"Starting Troy Email Fetcher API on {host}:{port}")
    logger.info(f"Email: {EMAIL_ADDRESS}")
    logger.info(f"Default Sender: {DEFAULT_SENDER}")
    
    # Auto-start polling if enabled
    if POLLING_ENABLED and POLLING_AUTO_START:
        try:
            logger.info("Auto-starting email polling service...")
            poller = get_email_poller()
            if poller.start_polling():
                logger.info(f"Email polling started automatically (interval: {POLLING_INTERVAL} seconds)")
            else:
                logger.warning("Polling service was already running")
        except Exception as e:
            logger.error(f"Failed to auto-start polling service: {e}", exc_info=True)
            logger.warning("API will start without polling service")
    elif POLLING_ENABLED:
        logger.info("Polling is enabled but auto-start is disabled. Use POST /api/polling/start to start manually.")
    else:
        logger.info("Polling is disabled")
    
    app.run(host=host, port=port, debug=False)

